<?php

namespace App\Http\Controllers\Api\DsrServices;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\DsrServiceHotel;
use App\Models\DsrServiceFlight;
use App\Models\DsrServiceVisa;
use App\Models\DsrServiceCarHire;
use App\Models\DsrServiceAssist;
use App\Models\DsrServiceInsurance;
use App\Models\DsrServiceOther;
use App\Models\DsrServiceTour;
use App\Models\User;
use App\Models\DsrPassenger;
use App\Models\DsrSupplierSummary;
use App\Service\ActivityService;
use App\Service\DsrMiceService\SupplierSummary;

class DsrPassengerServiceController extends Controller
{
    // DSR Service Hotel Create
    public function dsrHotelCreateUpdate(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_id' => 'nullable|integer',
            'hotel_id' => 'nullable|integer',
            'hotel_name' => 'nullable|string',
            'room_id' => 'nullable|integer',
            'meal_id' => 'nullable|integer',
            'bed' => 'nullable|integer',
            'guest_ids' => 'nullable|string',
            'booking_date' => 'nullable|date',
            'check_in_date' => 'nullable|date',
            'check_out_date' => 'nullable|date',
            'gross_amount' => 'nullable|numeric',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;

            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id']??0;
            $obj['entry_type'] = $request['entry_type'];
            $obj['hotel_id'] = $request['hotel_id']??0;
            $obj['hotel_name'] = $request['hotel_name'];
            $obj['room_id'] = $request['room_id']??0;
            $obj['room_number'] = $request['room_number']??null;
            $obj['meal_id'] = $request['meal_id']??0;
            $obj['bed'] = $request['bed'];
            $obj['guest_ids'] = $request['guest_ids'];
            $obj['booking_date'] = $request['booking_date'];
            $obj['check_in_date'] = $request['check_in_date']??null;
            $obj['check_out_date'] = $request['check_out_date']??null;
            $obj['supplier_id'] = $request['supplier_id']??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['total_nights'] = $request['total_nights'];
            $obj['gross_amount'] = $request['gross_amount'];
            $obj['tax_percent'] = $request['tax_percent'];
            $obj['tax_amount'] = $request['tax_amount'];
            $obj['retain'] = $request['retain'];
            $obj['total_amount'] = $request['total_amount'];
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['upload_image'] = $request['upload_image'];
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount'];
            $obj['charges'] = $request['charges'];
            $obj['confirmation_no'] = $request['confirmation_no'];
            $obj['occupancy_id'] = $request['occupancy_id'];
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;
            $obj['booking_confirmation_no'] = $request['booking_confirmation_no']??null;

            if ($request['id'] > 0) {
                $hotel = DsrServiceHotel::findOrFail($request['id']);

                //Reverse the amount
                $dsr_passenger = DsrPassenger::where('id', $hotel['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($hotel['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Adding the new Amount
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                //Delete DsrSupplierSummary
                DsrSupplierSummary::where('dsr_id', $request['dsr_id'])
                    ->where('transaction_id', $hotel['id'])
                    ->where('service_name', 'hotel')
                    ->delete();

                //Update Hotel
                $hotel->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Hotel', $hotel['id'], 'Updated');
            } else {
                // Create the DSR Service Hotel
                $hotel = DsrServiceHotel::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Hotel', $hotel['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$hotel,'hotel');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Hotel Updated Successfully' : 'DSR Service Hotel Created Successfully',
                'data' => $hotel
            ];

            return response()->json($response);
        }
    }
     // Cancel a DSR Service Hotel entry
    public function dsrHotelCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_hotel_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {

            $hotel = DsrServiceHotel::findOrFail($request['dsr_hotel_id']);
            $hotel->update(['is_cancle' => 1]);

            // $dsr_passenger=DsrPassenger::where('id',$hotel['passenger_id'])->first();
            // $dsr_passenger['service_total']=intval($dsr_passenger['service_total']) - intval($hotel['total_amount'])??0;
            // $dsr_passenger->update();

            $dsr_passenger = DsrPassenger::where('id', $hotel['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($hotel['total_amount'] ?? 0));

            $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $hotel['dsr_id'])
                ->where('transaction_id', $hotel['id'])
                ->where('service_name', 'hotel')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Hotel', $hotel['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Hotel Cancelled Successfully',
                'data' => $hotel
            ]);
        }      
    }

    // DSR Service Flight Create
    public function dsrFlightCreateUpdate(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_id' => 'nullable|integer',
            'travel_date' => 'nullable',
            'airline_name' => 'nullable|string|max:50',
            'guest_ids' => 'nullable|string',
            'travel_type_id' => 'nullable|integer',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;

            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id'];
            $obj['entry_type'] = $request['entry_type'];

            $obj['travel_date'] = $request['travel_date'];
            $obj['return_date'] = $request['return_date'];
            $obj['travel_type_id'] = $request['travel_type_id']??0;
            $obj['airline_name'] = $request['airline_name'];
            $obj['air_line_code'] = $request['air_line_code'];
            $obj['airline_digitcode'] = $request['airline_digitcode']??0;
            $obj['guest_ids'] = $request['guest_ids'];
            $obj['gds_portal'] = $request['gds_portal'];
            $obj['ticket_no'] = $request['ticket_no'];
            $obj['prn_no'] = $request['prn_no'];
            $obj['route'] = $request['route'];
            $obj['booking_date'] = $request['booking_date'];
            $obj['destination_code'] = $request['destination_code']??0;    //from the user masters
            $obj['multi_city'] = $request['multi_city']??null;

            $obj['from_date'] = $request['from_date']??null;
            $obj['to_date'] = $request['to_date']??null;
            $obj['booking_date'] = $request['booking_date']??null;
            $obj['booking_reference'] = $request['booking_reference']??null;
            $obj['agent_office_id'] = $request['agent_office_id']??null;
            $obj['flight_segment'] = $request['flight_segment']??null;
            
            $obj['supplier_id'] = $request['supplier_id']??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['gross_amount'] = $request['gross_amount']??0;
            $obj['tax_percent'] = $request['tax_percent'];
            $obj['tax_amount'] = $request['tax_amount'];
            $obj['retain'] = $request['retain'];
            $obj['total_amount'] = $request['total_amount'];
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['upload_image'] = $request['upload_image'];
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount'];
            $obj['charges'] = $request['charges'];
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $hotel = DsrServiceFlight::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $hotel['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($hotel['total_amount'] ?? 0));

                $dsr_passenger->update();


                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $hotel['dsr_id'])
                    ->where('transaction_id', $hotel['id'])
                    ->where('service_name', 'flight')
                    ->delete();

                $hotel->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Flight', $hotel['id'], 'Updated');
            } else {
                // Create the DSR Service Hotel
                $hotel = DsrServiceFlight::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Flight', $hotel['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$hotel,'flight');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Flight Updated Successfully' : 'DSR Service Flight Created Successfully',
                'data' => $hotel
            ];

            return response()->json($response);
        }
    }
     // Cancel a DSR Service Flight entry
    public function dsrFlightCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_flight_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {

            $hotel = DsrServiceFlight::findOrFail($request['dsr_flight_id']);
            $hotel->update(['is_cancle' => 1]);

            // $dsr_passenger=DsrPassenger::where('id',$hotel['passenger_id'])->first();
            // $dsr_passenger['service_total']=intval($dsr_passenger['service_total']) - intval($hotel['total_amount'])??0;
            // $dsr_passenger->update();

            $dsr_passenger = DsrPassenger::where('id', $hotel['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($hotel['total_amount'] ?? 0));

                $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $hotel['dsr_id'])
                ->where('transaction_id', $hotel['id'])
                ->where('service_name', 'flight')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Flight', $hotel['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Flight Cancelled Successfully',
                'data' => $hotel
            ]);
        }      
    }

    // DSR flight list
    public function dsrFlightList(Request $request){
        $dsr_flights = DsrServiceFlight::where('is_cancle',0)
            ->with('dsr')
            ->where('flight_segment','!=',null);

        if($request['custom_date']!=null && isset($request['custom_date']) && $request['custom_date']!=""){
            $dsr_flights = $dsr_flights->where('booking_date','=',$request['custom_date']);
        }

        if (!empty($request['from_date'])) {
            $from_date = date('Y-m-d', strtotime($request['from_date']));

            if (!empty($request['to_date'])) {
                // include the full to_date (end of day)
                $to_date = date('Y-m-d', strtotime($request['to_date'] . ' +1 day'));
            } else {
                // default to from_date + 1 day
                $to_date = date('Y-m-d', strtotime($from_date . ' +1 day'));
            }

            $dsr_flights = $dsr_flights->whereBetween('booking_date', [$from_date, $to_date]);
        }

        if($request['search']!=null && isset($request['search']) && $request['search']!=""){
            $dsr_flights = $dsr_flights->where(function ($query) use ($request) {
                $query->where('supplier_name', 'like', '%' . $request['search'] . '%')
                    ->orWhereHas('dsr', function ($query) use ($request) {
                        $query->where('dsr_no', 'like', '%' . $request['search'] . '%');
                    });

            });
        }

        $dsr_count=$dsr_flights->count();

        $dsr_flights = $dsr_flights
            ->orderBy('id', 'desc')
            ->skip($request['noofrec']*($request['currentpage']-1))->take($request['noofrec']??100)
            ->get();

        return response()->json([
            'success' => true,
            'count' => $dsr_count,
            'message' => 'DSR Service Flight Lists',
            'data' => $dsr_flights
        ]);
    }

    // DSR Service Visa Create
    public function dsrVisaCreateUpdate(Request $request){
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_visa_id' => 'required|integer',
            'passenger_id' => 'nullable|integer',
            'guest_ids' => 'nullable|string',
            'visa_no' => 'nullable|string',
            'passport_no' => 'nullable|string',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;

            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id'];
            $obj['entry_type'] = $request['entry_type'];

            $obj['passenger_visa_id'] = $request['passenger_visa_id']??0;
            $obj['passport_no'] = $request['passport_no'];
            $obj['dob'] = $request['dob'];
            $obj['guest_ids'] = $request['guest_ids'];
            $obj['email'] = $request['email'];
            $obj['visa_no'] = $request['visa_no'];
            $obj['visa_type_id'] = $request['visa_type_id'];       //from the masters
            $obj['visa_entry_type'] = $request['visa_entry_type'];
            // $obj['country_id'] = $request['country_id']??0;
            $obj['country_name'] = $request['country_name']??0;
            $obj['issue_date'] = $request['issue_date'];
            $obj['expiry_date'] = $request['expiry_date'];
            $obj['days'] = $request['days']??0;
            $obj['currency'] = $request['currency']??null;
            $obj['currency_symbol'] = $request['currency_symbol']??null;
            $obj['visa_entry_name'] = $request['visa_entry_name']??null;
            $obj['visa_type_name'] = $request['visa_type_name']??null;

            $obj['supplier_id'] = $request['supplier_id']??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['gross_amount'] = $request['gross_amount']??0;
            $obj['tax_percent'] = $request['tax_percent']??0;
            $obj['tax_amount'] = $request['tax_amount']??0;
            $obj['retain'] = $request['retain']??0;
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['total_amount'] = $request['total_amount']??0;
            $obj['upload_image'] = $request['upload_image'];
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount']??0;
            $obj['charges'] = $request['charges'];
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $hotel = DsrServiceVisa::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $hotel['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($hotel['total_amount'] ?? 0));

                $dsr_passenger->update();

                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $hotel['dsr_id'])
                    ->where('transaction_id', $hotel['id'])
                    ->where('service_name', 'visa')
                    ->delete();

                $hotel->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Visa', $hotel['id'], 'Updated');
            } else {

                //Duplicate visa no check
                $duplicate = DsrServiceVisa::where('visa_no', $obj['visa_no'])
                    ->where('dsr_id', $obj['dsr_id'])
                    ->where('passenger_id', $obj['passenger_id'])
                    ->first();

                if ($duplicate) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Visa No already exists'
                    ]);
                }


                // Create the DSR Service Hotel
                $hotel = DsrServiceVisa::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Visa', $hotel['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$hotel,'visa');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Visa Updated Successfully' : 'DSR Service Visa Created Successfully',
                'data' => $hotel
            ];

            return response()->json($response);
        }
    }
    // Cancel a DSR Service Visa entry
    public function dsrVisaCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_visa_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {

            $visa = DsrServiceVisa::findOrFail($request['dsr_visa_id']);
            $visa->update(['is_cancle' => 1]);

            // $dsr_passenger=DsrPassenger::where('id',$visa['passenger_id'])->first();
            // $dsr_passenger['service_total']=intval($dsr_passenger['service_total']) - intval($visa['total_amount'])??0;
            // $dsr_passenger->update();

            $dsr_passenger = DsrPassenger::where('id', $visa['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($visa['total_amount'] ?? 0));

            $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $visa['dsr_id'])
                ->where('transaction_id', $visa['id'])
                ->where('service_name', 'visa')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Visa', $visa['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Visa Cancelled Successfully',
                'data' => $visa
            ]);
        }      
    }

    // DSR Service Car Hire Create
    public function dsrCarHireCreateUpdate(Request $request){
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'vehicle_type_id' => 'required|integer',
            'pickup_location' => 'required',
            'drop_location' => 'required',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);
 
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;
    
            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id'];
            $obj['entry_type'] = $request['entry_type'];
 
            $obj['vehicle_type_id'] = $request['vehicle_type_id']??0;     //from the masters
            $obj['vehicle_type_name'] = $request['vehicle_type_name']??null;     //from the masters
            $obj['days'] = $request['days']??0;
            $obj['pickup_location'] = $request['pickup_location']??null;
            $obj['drop_location'] = $request['drop_location']??null;
            $obj['guest_ids'] = $request['guest_ids']??null;
 
            $obj['supplier_id'] = $request['supplier_id']??0??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['gross_amount'] = $request['gross_amount']??0.0;         //car_hire_price
            $obj['tax_percent'] = $request['tax_percent']??0.0;
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['tax_amount'] = $request['tax_amount']??0.0;
            $obj['retain'] = $request['retain']??0.0;
            $obj['total_amount'] = $request['total_amount']??0.0;
            $obj['upload_image'] = $request['upload_image']??null;
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount']??0.0;
            $obj['charges'] = $request['charges']??null;
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $car_hire = DsrServiceCarHire::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $car_hire['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($car_hire['total_amount'] ?? 0));

                $dsr_passenger->update();

                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $car_hire['dsr_id'])
                    ->where('transaction_id', $car_hire['id'])
                    ->where('service_name', 'car_hire')
                    ->delete();

                $car_hire->update($obj);
                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Car Hire', $car_hire['id'], 'Updated');
            } else {
                // Create the DSR Service Hotel
                $car_hire = DsrServiceCarHire::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Car Hire', $car_hire['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$car_hire,'car_hire');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Car Hire Updated Successfully' : 'DSR Service Car Hire Created Successfully',
                'data' => $car_hire
            ];

            return response()->json($response);
        }
    }
    // Cancel a DSR Service Car Hire entry
    public function dsrCarHireCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_car_hire_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {
 
            $car_hire = DsrServiceCarHire::findOrFail($request['dsr_car_hire_id']);
            $car_hire->update(['is_cancle' => 1]);

            $dsr_passenger = DsrPassenger::where('id', $car_hire['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($car_hire['total_amount'] ?? 0));

            $dsr_passenger->update();

             // Delete Supplier Summary
             DsrSupplierSummary::where('dsr_id', $car_hire['dsr_id'])
                ->where('transaction_id', $car_hire['id'])
                ->where('service_name', 'car_hire')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Car Hire', $car_hire['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Car Hire Cancelled Successfully',
                'data' => $car_hire
            ]);
        }      
    }

    // DSR Service Assists Create
    public function dsrAsistsCreateUpdate(Request $request){
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_id' => 'required|integer',
            'airport_name' => 'required',
            'no_of_persons' => 'required',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);
 
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;
    
            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id'];
            $obj['entry_type'] = $request['entry_type'];
 
            $obj['airport_id'] = $request['airport_id']??0;     //from the masters
            $obj['airport_name'] = $request['airport_name'];     //from the masters
            $obj['meeting_type'] = $request['meeting_type']??null;
            $obj['no_of_persons'] = $request['no_of_persons']??0;
            $obj['guest_ids'] = $request['guest_ids']??null;
 
            $obj['supplier_id'] = $request['supplier_id']??0??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['gross_amount'] = $request['gross_amount']??0.0;         //Assists price
            $obj['tax_percent'] = $request['tax_percent']??0.0;
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['tax_amount'] = $request['tax_amount']??0.0;
            $obj['retain'] = $request['retain']??0.0;
            $obj['total_amount'] = $request['total_amount']??0.0;
            $obj['upload_image'] = $request['upload_image']??null;
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount']??0.0;
            $obj['charges'] = $request['charges']??null;
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $assists = DsrServiceAssist::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $assists['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($assists['total_amount'] ?? 0));

                $dsr_passenger->update();

                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $assists['dsr_id'])
                    ->where('transaction_id', $assists['id'])
                    ->where('service_name', 'assist')
                    ->delete();

                $assists->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Assists', $assists['id'], 'Updated');
            } else {
                // Create the DSR Service Hotel
                $assists = DsrServiceAssist::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Assists', $assists['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$assists,'assist');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Assists Updated Successfully' : 'DSR Service Assists Created Successfully',
                'data' => $assists
            ];

            return response()->json($response);
        }
    }
    // Cancel a DSR Service Assists entry
    public function dsrAsistsCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_assists_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {
 
            $assists = DsrServiceAssist::findOrFail($request['dsr_assists_id']);
            $assists->update(['is_cancle' => 1]);

            $dsr_passenger = DsrPassenger::where('id', $assists['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($assists['total_amount'] ?? 0));

            $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $assists['dsr_id'])
                ->where('transaction_id', $assists['id'])
                ->where('service_name', 'assist')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Assists', $assists['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Assists Cancelled Successfully',
                'data' => $assists
            ]);
        }      
    }

     // DSR Service Insurance Create
    public function dsrInsuranceCreateUpdate(Request $request){
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_id' => 'required|integer',
            'passenger_insurance_id' => 'required|integer',
            'policy_no' => 'required',
            'insurance_plan' => 'required',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);
 
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;
    
            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id'];
            $obj['entry_type'] = $request['entry_type'];
 
            $obj['passenger_insurance_id'] = $request['passenger_insurance_id']??0;     //from the masters
            $obj['company_name'] = $request['company_name'];     //from the masters
            $obj['policy_no'] = $request['policy_no']??null;
            $obj['insurance_plan'] = $request['insurance_plan']??0;
            $obj['expiry_date'] = $request['expiry_date']??null;
            $obj['issue_date'] = $request['issue_date']??null;
 
            $obj['supplier_id'] = $request['supplier_id']??0??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['gross_amount'] = $request['gross_amount']??0.0;         //Insurance price
            $obj['tax_percent'] = $request['tax_percent']??0.0;
            $obj['tax_amount'] = $request['tax_amount']??0.0;
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['retain'] = $request['retain']??0.0;
            $obj['total_amount'] = $request['total_amount']??0.0;
            $obj['upload_image'] = $request['upload_image']??null;
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount']??0.0;
            $obj['charges'] = $request['charges']??null;
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $insurance = DsrServiceInsurance::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $insurance['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($insurance['total_amount'] ?? 0));

                $dsr_passenger->update();

                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $insurance['dsr_id'])
                    ->where('transaction_id', $insurance['id'])
                    ->where('service_name', 'insurance')
                    ->delete();

                $insurance->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Insurance', $insurance['id'], 'Updated');
            } else {

                //Check if the Policy No already exists
                $check = DsrServiceInsurance::where('policy_no', $obj['policy_no'])->where('dsr_id', $obj['dsr_id'])->first();

                if ($check) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Policy No already exists'

                    ]);
                }

                // Create the DSR Service Hotel
                $insurance = DsrServiceInsurance::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Insurance', $insurance['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$insurance,'insurance');
            }


            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Insurance Updated Successfully' : 'DSR Service Insurance Created Successfully',
                'data' => $insurance
            ];

            return response()->json($response);
        }
    }
    // Cancel a DSR Service Insurance entry
    public function dsrInsuranceCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_insurance_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {
 
            $insurance = DsrServiceInsurance::findOrFail($request['dsr_insurance_id']);
            $insurance->update(['is_cancle' => 1]);

            $dsr_passenger = DsrPassenger::where('id', $insurance['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($insurance['total_amount'] ?? 0));

            $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $insurance['dsr_id'])
                ->where('transaction_id', $insurance['id'])
                ->where('service_name', 'insurance')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Insurance', $insurance['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Insurance Cancelled Successfully',
                'data' => $insurance
            ]);
        }      
    }

     // DSR Service Others Create
    public function dsrOthersCreateUpdate(Request $request){
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_id' => 'required|integer',
            'item_id' => 'required|integer',
            'quantity' => 'required',
            'price' => 'required',
            'tax_percent' => 'nullable|numeric',
            'tax_amount' => 'nullable|numeric',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);
 
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        } else {
            // Activity Service
            $activity = new ActivityService;
    
            $obj = [];
            $obj['dsr_id'] = $request['dsr_id'];
            $obj['passenger_id'] = $request['passenger_id'];
            $obj['entry_type'] = $request['entry_type'];
 
            $obj['item_id'] = $request['item_id']??0;     //from the Item table
            $obj['item_name'] = $request['item_name'];   
            $obj['quantity'] = $request['quantity']??0;
            $obj['price'] = $request['price']??0;
 
            $obj['supplier_id'] = $request['supplier_id']??0??0;
            $obj['supplier_name'] = $request['supplier_name'];
            $obj['gross_amount'] = $request['gross_amount']??0.0;         // price
            $obj['tax_percent'] = $request['tax_percent']??0.0;
            $obj['tax_amount'] = $request['tax_amount']??0.0;
            $obj['retain'] = $request['retain']??0.0;
            $obj['supplier_total'] = $request['supplier_total']??0.0;
            $obj['total_amount'] = $request['total_amount']??0.0;
            $obj['upload_image'] = $request['upload_image']??null;
            $obj['note'] = $request['note'];
            $obj['discount'] = $request['discount']??0.0;
            $obj['charges'] = $request['charges']??null;
            $obj['supplier_amount'] = $request['supplier_amount']??0;
            $obj['supplier_tax'] = $request['supplier_tax']??0;
            $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $others = DsrServiceOther::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $others['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($others['total_amount'] ?? 0));

                $dsr_passenger->update();

                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $others['dsr_id'])
                    ->where('transaction_id', $others['id'])
                    ->where('service_name', 'others')
                    ->delete();

                $others->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Others', $others['id'], 'Updated');
            } else {
                // Create the DSR Service Hotel
                $others = DsrServiceOther::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service Others', $others['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$others,'others');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service Others Updated Successfully' : 'DSR Service Others Created Successfully',
                'data' => $others
            ];

            return response()->json($response);
        }
    }
    // Cancel a DSR Service Others entry
    public function dsrOthersCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_others_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {
 
            $others = DsrServiceOther::findOrFail($request['dsr_others_id']);
            $others->update(['is_cancle' => 1]);

            $dsr_passenger = DsrPassenger::where('id', $others['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($others['total_amount'] ?? 0));

            $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $others['dsr_id'])
                ->where('transaction_id', $others['id'])
                ->where('service_name', 'others')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Others', $others['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Others Cancelled Successfully',
                'data' => $others
            ]);
        }      
    }

    //get all details service by a pessenger id
    public function getDsrServiceHotelByPassengerId(Request $request)
    {
        $validator = validator($request->all(), [
            'passenger_id' => 'required|integer'
        ]);

        if ($validator->fails()) {

            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {

            $dsr_passenger=DsrPassenger::where('id',$request['passenger_id'])
                ->with('passenger','dsr:id,dsr_no')
                ->first();

            $dsrServiceHotel = DsrServiceHotel::where('passenger_id', $request['passenger_id'])
                ->with('hotel','room:id,room_number')
                ->where('is_cancle',0)
                ->orderBy('id','desc')
                ->get();

                foreach ($dsrServiceHotel as $key => $hotel) {
                    $guestIds = explode(',', $hotel['guest_ids'] ?? '');
                    $hotel['guests']=User::whereIn('id',$guestIds)->get();
                }

            $dsrServiceFlight = DsrServiceFlight::where('passenger_id', $request['passenger_id'])
                ->where('is_cancle',0)
                ->orderBy('id','desc')
                ->get();

                foreach ($dsrServiceFlight as $key => $flight) {
                    $guestIds = explode(',', $flight['guest_ids'] ?? '');
                    $flight['guests']=User::whereIn('id',$guestIds)->get();
                }

            $dsrServicVisa = DsrServiceVisa::where('passenger_id', $request['passenger_id'])
                ->where('is_cancle',0)
                ->with('visa_passenger:id,display_name','country:id,name')
                ->orderBy('id','desc')
                ->get();

                foreach ($dsrServicVisa as $key => $visa) {
                    $guestIds = explode(',', $visa['guest_ids'] ?? '');
                    $visa['guests']=User::whereIn('id',$guestIds)->get();
                }

            $dsrServicAssist = DsrServiceAssist::where('passenger_id', $request['passenger_id'])
                ->where('is_cancle',0)
                ->orderBy('id','desc')
                ->get();

                foreach ($dsrServicAssist as $key => $assist) {
                    $guestIds = explode(',', $assist['guest_ids'] ?? '');
                    $assist['guests']=User::whereIn('id',$guestIds)->get();
                }

            $dsrServicCarHire = DsrServiceCarHire::where('passenger_id', $request['passenger_id'])
                ->where('is_cancle',0)
                ->orderBy('id','desc')
                ->get();

                foreach ($dsrServicCarHire as $key => $car_hire) {
                    $guestIds = explode(',', $car_hire['guest_ids'] ?? '');
                    $car_hire['guests']=User::whereIn('id',$guestIds)->get();
                }

            $dsrServicInsurance = DsrServiceInsurance::where('passenger_id', $request['passenger_id'])
                ->where('is_cancle',0)
                ->with('passenger')
                ->orderBy('id','desc')
                ->get();


            $dsrServicOthers = DsrServiceOther::where('passenger_id', $request['passenger_id'])
                ->where('is_cancle',0)
                ->orderBy('id','desc')
                ->get();

            $dsrServicTour = DsrServiceTour::where('passenger_id', $request['passenger_id'])
                ->with('tour_package')
                ->where('is_cancle',0)
                ->orderBy('id','desc')
                ->get();

            foreach ($dsrServicTour as $key => $tour) {
                $guestIds = explode(',', $tour['guest_ids'] ?? '');
                $tour['guests']=User::whereIn('id',$guestIds)->get();
            }

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Passenger Details fetched',
                'dsr_passenger'=>$dsr_passenger,
                'dsr_hotel' => $dsrServiceHotel,
                'dsr_flight' => $dsrServiceFlight,
                'dsr_visa' => $dsrServicVisa,
                'dsr_assist' => $dsrServicAssist,
                'dsr_insurance' => $dsrServicInsurance,
                'dsr_others' => $dsrServicOthers,
                'dsr_car_hire' => $dsrServicCarHire,
                'dsr_tour' => $dsrServicTour
            ]);
            
        }
    }

    // Tour Package Details
    public function tourPackageController(Request $request){

        $validator = validator($request->all(), [
            'dsr_id' => 'required|integer',
            'passenger_id' => 'required|integer',
            'tour_package_id'=>'required|integer',
            'days'=>'required',
            'retain' => 'nullable|numeric',
            'total_amount' => 'nullable|numeric'
        ]);

        // Activity Service
            $activity = new ActivityService;

        $obj['dsr_id'] = $request['dsr_id'];
        $obj['passenger_id'] = $request['passenger_id'];
        $obj['tour_package_id'] = $request['tour_package_id'];
        $obj['package_name'] = $request['package_name'];
        $obj['hotel_type'] = $request['hotel_type'];
        $obj['meal_id'] = $request['meal_id']??0;
        $obj['is_transport'] = $request['is_transport']??0;
        $obj['days'] = $request['days'];
        $obj['destination'] = $request['destination'];

        $obj['guest_ids'] = $request['guest_ids']??null;
 
        $obj['supplier_id'] = $request['supplier_id']??0??0;
        $obj['supplier_name'] = $request['supplier_name'];
        $obj['gross_amount'] = $request['gross_amount']??0.0;         //car_hire_price
        $obj['tax_percent'] = $request['tax_percent']??0.0;
        $obj['supplier_total'] = $request['supplier_total']??0.0;
        $obj['tax_amount'] = $request['tax_amount']??0.0;
        $obj['retain'] = $request['retain']??0.0;
        $obj['total_amount'] = $request['total_amount']??0.0;
        $obj['upload_image'] = $request['upload_image']??null;
        $obj['note'] = $request['note'];
        $obj['discount'] = $request['discount']??0.0;
        $obj['charges'] = $request['charges']??null;
        $obj['supplier_amount'] = $request['supplier_amount']??0;
        $obj['supplier_tax'] = $request['supplier_tax']??0;
        $obj['customer_amount'] = $request['customer_amount']??0;

            if ($request['id'] > 0) {
                $tour = DsrServiceTour::findOrFail($request['id']);

                $dsr_passenger = DsrPassenger::where('id', $tour['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($tour['total_amount'] ?? 0));

                $dsr_passenger->update();

                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Delete Supplier Summary
                DsrSupplierSummary::where('dsr_id', $tour['dsr_id'])
                    ->where('transaction_id', $tour['id'])
                    ->where('service_name', 'tour_package')
                    ->delete();

                $tour->update($obj);

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service tour package', $tour['id'], 'Updated');
            } else {
                // Create the DSR Service Hotel
                $tour = DsrServiceTour::create($obj);

                $dsr_passenger = DsrPassenger::where('id', $request['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    + ((float) ($obj['total_amount'] ?? 0));

                $dsr_passenger->update();

                // Creating Activities
                $activity->activityCreateUpdate('DSR Service tour package', $tour['id'], 'Created');
            }

            // Creating supplier entry summary
            if($request['supplier_id'] > 0){
                $supplier_summary = new SupplierSummary;
                $supplier_summary->dsrSupplierSummaryCreate($request,$tour,'tour_package');
            }

            $response = [
                'success' => true,
                'message' => $request['id'] > 0 ? 'DSR Service tour package Updated Successfully' : 'DSR Service tour package Created Successfully',
                'data' => $tour
            ];

            return response()->json($response);
        
    }

     // Cancel a DSR Service Insurance entry
    public function dsrTourPackageCancel(Request $request)
    {
        // Validate the input
        $validator = validator($request->all(), [
            'dsr_tour_package_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);

        } else {
 
            $insurance = DsrServiceTour::findOrFail($request['dsr_tour_package_id']);
            $insurance->update(['is_cancle' => 1]);

            $dsr_passenger = DsrPassenger::where('id', $insurance['passenger_id'])->first();
                $dsr_passenger['service_total'] = (float) $dsr_passenger['service_total'] 
                    - ((float) ($insurance['total_amount'] ?? 0));

            $dsr_passenger->update();

            // Delete Supplier Summary
            DsrSupplierSummary::where('dsr_id', $insurance['dsr_id'])
                ->where('transaction_id', $insurance['id'])
                ->where('service_name', 'tour_package')
                ->delete();

            // Activity Service
            $activity = new ActivityService;
            $activity->activityCreateUpdate('DSR Service Insurance', $insurance['id'], 'Cancelled');

            return response()->json([
                'success' => true,
                'message' => 'DSR Service Tour Package Cancelled Successfully',
                'data' => $insurance
            ]);
        }      
    }

}
