<?php

namespace App\Http\Controllers\Api\Reports;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use DB;
use Carbon\Carbon;
use App\Models\Transaction;

class ProfitAndLossController extends Controller
{
    //Summary Report Of Profit and Loss
    public function getProfitAndLossSummaryReport(Request $request)
    {
        // ✅ Default: current month
        $startDate = $request->input('start_date') ?: Carbon::today()->subMonth()->format('Y-m-d') ;
        $endDate = $request->input('end_date') ?: Carbon::today()->format('Y-m-d');

        $heads = DB::table('financial_statement_heads')->where('is_deleted', '0')
            ->where('is_profit_loss', '1')
            ->get();

        $result = [];

        $incomeTotal   = 0;
        $expenseTotal  = 0;
        $cogsTotal     = 0;

        foreach ($heads as $head) {
             // ✅ Decode JSON string [14,36] → [14, 36]
            $accountIds = json_decode($head->account_ids, true);

            if (empty($accountIds)) {
                continue; // skip if no accounts mapped
            }

            $accounts = DB::table('transactions as t')
                ->join('accounts as a', 't.account_id', '=', 'a.id')
                ->select(
                    't.account_id',
                    'a.account_name',
                    'a.account_code',
                    // DB::raw('SUM(t.credit - t.debit) as total')
                    DB::raw('CAST(SUM(t.credit - t.debit) AS DECIMAL(15,2)) as total')
                )
                ->whereIn('t.account_id', $accountIds)
                ->whereBetween('t.transaction_date', [$startDate, $endDate])
                ->groupBy('t.account_id', 'a.account_name', 'a.account_code')
                ->get();

            $headTotal = $accounts->sum('total');

            // --- Classify head type ---
            $headName = strtolower($head->head_name);

            if (strpos($headName, 'income') !== false) {
                $incomeTotal += $headTotal;
            } elseif (strpos($headName, 'cogs') !== false || strpos($headName, 'cost of goods') !== false) {
                $cogsTotal += $headTotal;
            } else {
                $expenseTotal += $headTotal;
            }

            $result[] = [
                'head_name'  => $head->head_name,
                'accounts'   => $accounts,
                'head_total' => $headTotal
            ];
        }
        // ✅ Derived Totals
        $grossProfit = $incomeTotal + $cogsTotal; // cogsTotal is usually negative
        $netProfit   = $incomeTotal + $cogsTotal + $expenseTotal;

        return [
            'from_date' => $startDate,
            'to_date'   => $endDate,
            'report'    => $result,
            'overall'   => [
                'gross_profit'          => $grossProfit,
                'net_profit_before_tax' => $netProfit
            ]
        ];
    }
}
