<?php

namespace App\Http\Controllers\Api\cms;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Models\cms\PreArrivalDeparture;
use App\Models\cms\Order;
use App\Models\cms\ContainerOwner;
use App\Models\cms\LogisticProvider;
use App\Models\cms\Yard;
use App\Models\cms\Masters\ContainerType;
use Illuminate\Support\Facades\Session;
use App\Service\cms\rderService;
use App\Models\cms\ContainerDetail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Log;
use DB;

class ApprovalController extends Controller
{
    public function index(Request $request)
    {
        try {
            $user = Auth::user();

            // Fetch pre-arrival records with conditions
            $pre_arrival = PreArrivalDeparture::where('approved', 0)
                ->where('order_type', 'arrival')
                ->where('status', 1)
                ->with('orders', 'orders.container_owner:id,company_name', 'entrybyid');

                // Filtering based on 'date'
            if ($request->custom_date) {
                $pre_arrival->whereDate('date', $request->custom_date);
            }

             // Filtering based on from date to date
            if (isset($request->fromDate) && isset($request->toDate)) {
                $pre_arrival->whereBetween('date', [$request->fromDate, $request->toDate]);
            }

            // Searching based on 'search' keyword ( container_id,)
           if (!empty($request->search)) {
                $search = $request->search;
                $pre_arrival->where(function ($query) use ($search) {
                    $query->whereExists(function ($subQuery) use ($search) {
                        $subQuery->select(DB::raw(1))
                            ->from('container_details')
                            ->whereRaw('FIND_IN_SET(container_details.id, pre_arrival_departure.container_detail_ids)')
                            ->where('container_details.container_id', 'like', '%' . $search . '%');
                    });
                });
            }


            // Count total records before pagination
            $count = $pre_arrival->count();

            // Retrieve the data
            $pre_arrival = $pre_arrival->orderBy('id', 'desc')
                ->get();

            // Add container details for each pre-arrival record
            foreach ($pre_arrival as $container) {
                if (!empty($container->container_detail_ids)) {
                    $container_details = ContainerDetail::whereIn('id', explode(',', $container->container_detail_ids))
                        ->select('id', 'container_id')
                        ->get();

                    $container['container_details'] = $container_details;
                } else {
                    $container['container_details'] = collect();
                }
            }

            // Return JSON response with the required structure
            return response()->json([
                'success' => true,
                'total' => $count,
                'data' => $pre_arrival,
            ], 200);
        } catch (\Exception $e) {
            // Handle unexpected errors
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }



    public function details(Request $request)
    {
        try {
            // Validate the request
            $validator = Validator::make($request->all(), [
                'id' => 'required', // Ensure ID is an integer
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data' => null
                ], 422);
            }

            // Fetch the pre-arrival record with related block and documents
            $pre_arrival = PreArrivalDeparture::with([
                    'block:id,block_name','ship_details','port_details','vendor','yard','logistic_provider','vehicle','driver','items'
                ])
                ->findOrFail($request->id);

            // Ensure container_detail_ids is not null
            $containerDetailIds = $pre_arrival->container_detail_ids
                ? explode(',', $pre_arrival->container_detail_ids)
                : [];

            // Get container details with related data
            $container_details = ContainerDetail::whereIn('id', $containerDetailIds)
                ->with([
                    // 'images' => function ($query) {
                    //     $query->where('order_type', 'arrival');
                    // },
                    'container_owner',
                    'container_types',
                    'logisticProvider',
                    'vehicleMaster',
                    'yard'
                ])
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Pre-arrival details fetched successfully',
                'data' => [
                    'pre_arrival' => $pre_arrival,
                    'container_details' => $container_details
                ]
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Pre-arrival record not found',
                'data' => null
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An unexpected error occurred',
                'error' => $e->getMessage(),
                'data' => null
            ], 500);
        }
    }




    public function sendForArrivalApproval(Request $request)
    {
        try {
            // ✅ Validate the request
            $validator = Validator::make($request->all(), [
                'id' => 'required|integer', // Ensure ID is required and must be an integer
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data' => null
                ], 200);
            }

            // ✅ Fetch user
            $user = Auth::user();

            // ✅ Find the record
            $arrival = PreArrivalDeparture::find($request->id);

            if (!$arrival) {
                return response()->json([
                    'success' => false,
                    'message' => 'Arrival record not found',
                    'data' => null
                ], 200);
            }

            // ✅ Update record
            $arrival->status = 1;
            // $arrival->reporting_id = $user->reporting_id;
            $arrival->save();

            // ✅ Return JSON response
            return response()->json([
                'success' => true,
                'message' => 'Approved Successfully.',
                'data' => $arrival
            ], 200);
        } catch (\Exception $e) {
            // Handle unexpected errors
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    public function edit($id)
    {
        $pre_arrival = PreArrivalDeparture::with('documents')->where('id', $id)->first();

        $container_details = ContainerDetail::whereIn('id', explode(',', $pre_arrival['container_detail_ids']))
            ->with(['images' => function ($query) {
                $query->where('order_type', 'arrival');
            }])
            ->get();
        // dd($logisticProvider);
        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();


        return view('pre-arrival.edit', compact('pre_arrival', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster', 'container_details'));
    }


    public function show($id)
    {
        $arrival = PreArrivalDeparture::with('block:id,block_name', 'documents')
            ->where('id', $id)->first();

        $container_details = ContainerDetail::whereIn('id', explode(',', $arrival['container_detail_ids']))
            ->with(['images' => function ($query) {
                $query->where('order_type', 'arrival');
            }])
            ->get();
        // dd($orders);
        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $yards = Yard::where('status', 0)->get();

        return view('arrival-approval.show', compact('arrival', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster', 'yards', 'container_details'));
    }

    public function approveArrival(Request $request)
    {
        $today = new Carbon();

        // Validate the request
        $validator = Validator::make($request->all(), [
            'id' => 'required', // Ensure ID is an integer
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'data' => null
            ], 200);
        }

        // Update Approved
        $arrival = PreArrivalDeparture::where('id', $request['id'])->first();
        $arrival['approved'] = 1;
        $arrival->update();

        $generateAidno = $this->generateRandomPID();

        if ($arrival['container_detail_ids'] != null) {

            foreach (explode(',', $arrival['container_detail_ids']) as $key => $value) {

                Log::info($value);

                $orders = Order::create([
                    'date' => $today->format('Y-m-d'),
                    'pid_no' => $generateAidno
                ]);
                $orders['arrival_id'] = $arrival['id'];
                $orders['order_id'] = 'OID' . $orders['id'];
                $orders['container_id'] = $value ??null;
                $orders->update();

                Log::info($orders);

                $container = ContainerDetail::where('id', $value)->lockForUpdate()->first();
                $container['yard_id'] = $arrival['yard_id'];
                $container['block_id'] = $arrival['block_id'];
                $container['order_id'] = $orders['id'];
                $container['pid_no'] = $generateAidno;
                $container->update();

                Log::info("Container Updated");
            }
        }

        // Update the container type availability
        // $orderservice = new OrderService;

        // $container_details = ContainerDetail::whereIn('id', explode(',', $arrival['container_detail_ids']))->get();

        // foreach ($container_details as $key => $details) {
        //     if ($arrival['block_id'] != null) {
        //         $orderservice->availableContainertype($arrival['block_id'], $details, 1);
        //     }
        // }

        $response = [
            'success' => true,
            'message' => 'Approved Successfully.',
            'data' => $arrival
        ];
        return response()->json($response, 200);
    }

    // Reject Arrival Approval
    public function rejectArrivalApproval(Request $request)
    {
        $today = new Carbon();

        // Validate the request
        $validator = Validator::make($request->all(), [
            'id' => 'required', // Ensure ID is an integer
            'reject_reason' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'data' => null
            ], 422);
        }


        // Update Approved
        $arrival = PreArrivalDeparture::where('id', $request['id'])->first();
        $arrival['status'] = 0;
        $arrival['reject_reason'] = $request['reject_reason'];
        $arrival->update();

        $response = [
            'success' => true,
            'message' => 'Approved Successfully.',
            'data' => $arrival
        ];
        return response()->json($response, 200);
    }

    private function generateRandomPID()
    {

        $unique = false;

        while (!$unique) {
            $random_string = 'AID' . rand(10000, 99999);
            $count = Order::where('pid_no', $random_string)->count();
            if ($count == 0) {
                $unique = true;
            }
        }

        return $random_string;
    }

    public function filter(Request $request)
    {
        $pre_arrival = PreArrivalDeparture::where('approved', 0)->where('order_type', 'arrival')
            ->with('orders', 'orders.container_owner:id,company_name');

        if ($request['date'] != '' && isset($request['date'])) {
            $pre_arrival = $pre_arrival->whereDate('date', $request['date']);
        }
        if ($request['container_id'] != '' && isset($request['container_id'])) {
            $pre_arrival = $pre_arrival->where('container_id', $request['container_id']);
        }
        if ($request['container_status'] != '' && isset($request['container_status'])) {
            // $pre_arrival=$pre_arrival->where('container_status',$request['container_status']);
            $pre_arrival = $pre_arrival->whereHas('orders', function ($query) use ($request) {
                $query->where('container_status', $request['container_status']);
            });
        }
        if ($request['container_condition'] != '' && isset($request['container_condition'])) {
            $pre_arrival = $pre_arrival->whereHas('orders', function ($query) use ($request) {
                $query->where('container_condition', $request['container_condition']);
            });
        }

        $pre_arrival = $pre_arrival->get();

        return view('pre-arrival.index', compact('pre_arrival'));
    }

    // public function sendForArrivalApproval($id)
    // {
    //     // dd($id);
    //     $user = Auth::user();

    //     $arrival = PreArrivalDeparture::where('id', $id)->first();
    //     $arrival['status'] = 1;
    //     $arrival['reporting_id'] = $user['reporting_id'];
    //     $arrival->update();

    //     return redirect()->route('arrival.index')->with('success', 'Approved Successfully.');
    // }
}
