<?php

namespace App\Http\Controllers\Api\cms;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Str;
use Carbon\Carbon;
use App\Models\cms\PlacedDepartureOrder;
use App\Models\cms\Order;
use App\Models\cms\ContainerOwner;
use App\Models\cms\LogisticProvider;
use App\Models\cms\Yard;
use App\Models\cms\ContainerDetail;
use App\Models\cms\Masters\ContainerType;
use App\Models\cms\Masters\VehicleType;
use App\Service\cms\OrderService;
use App\Service\Trip\TripService;
use App\Service\SequenceService;
use Illuminate\Support\Facades\Session;
use Auth;
use DB;

class ApprovalDepartureController extends Controller
{
    public function index(Request $request)
    {
        try {
            // Get the authenticated user
            $user = Auth::user();

            // Fetch departure records with conditions
            $pre_arrival = PlacedDepartureOrder::where('approved', 0)
                ->where('order_type', 'departure')
                ->where('status', 1)
                ->with('orders', 'orders.container_owner:id,company_name', 'entrybyid');

            if (!empty($request->custom_date)) {
                $pre_arrival->whereDate('date', $request->custom_date);
            }

             // Filtering based on from date to date
            if (!empty($request->fromDate) && !empty($request->toDate)) {
                $pre_arrival->whereBetween('date', [$request->fromDate, $request->toDate]);
            }

            // Searching based on 'search' keyword ( container_id,)
           if (!empty($request->search)) {
                $search = $request->search;
                $pre_arrival->where(function ($query) use ($search) {
                    $query->whereExists(function ($subQuery) use ($search) {
                        $subQuery->select(DB::raw(1))
                            ->from('container_details')
                            ->whereRaw('FIND_IN_SET(container_details.id, placed_departure_order.container_detail_ids)')
                            ->where('container_details.container_id', 'like', '%' . $search . '%');
                    });
                });
            }

            // Count total records before retrieving data
            $count = $pre_arrival->count();

            // Retrieve the data
            $pre_arrival = $pre_arrival->orderBy('id', 'desc')
                ->get();

            // Add container details for each departure record
            foreach ($pre_arrival as $container) {
                if (!empty($container->container_detail_ids)) {
                    $container_details = ContainerDetail::whereIn('id', explode(',', $container->container_detail_ids))
                        ->select('id', 'container_id')
                        ->get();

                    $container['container_details'] = $container_details;
                } else {
                    $container['container_details'] = collect(); // Empty collection if no container details
                }
            }

            // Return JSON response with structured output
            return response()->json([
                'success' => true,
                'total' => $count,
                'data' => $pre_arrival,
            ], 200);
        } catch (\Exception $e) {
            // Handle unexpected errors and return structured error response
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function details(Request $request)
    {
        try {
            // Validate the request
            $validator = validator($request->all(), [
                'id' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            // Fetch the departure record with related orders 
            $departures = PlacedDepartureOrder::with('orders','logistic_provider','vendor','driver','vehicle')
                ->findOrFail($request->id);

            // Get container details filtered by 'departure' order type

                $container_details = ContainerDetail::whereIn('id', explode(',', $departures->container_detail_ids))
                ->with([
                    'container_owner',
                    'container_types',
                    'logisticProvider',
                    'vehicleMaster',
                    'yard'
                ])
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Pre-arrival details fetched successfully',
                'data' => [
                    'pre_arrival' => $departures,
                    'container_details' => $container_details
                ]
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // Handle case when the record is not found
            return response()->json([
                'success' => false,
                'message' => 'Departure record not found',
            ], 404);
        } catch (\Exception $e) {
            // Handle any unexpected errors
            return response()->json([
                'success' => false,
                'message' => 'An unexpected error occurred',
                'error' => $e->getMessage(),
            ], 200);
        }
    }

    public function approveDeparture(Request $request)
    {
        // Validate the request
        $validator = validator($request->all(), [
            'id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 200);
        }

        // Find the departure record by ID and approve it
        $departure = PlacedDepartureOrder::where('id', $request['id'])->with('orders', 'orders.arrival')->first();

        $departure['approved'] = 1;
        $departure->update();

        $generated_no = $this->generateRandomPID();

        // Update the container type availability
        $orderservice = new OrderService;

        $container_details = ContainerDetail::whereIn('id', explode(',', $departure['container_detail_ids']))->get();

        // Updating Container Details and Orders
        if ($container_details != null) {

            foreach ($container_details as $key => $details) {

                $order = Order::where('id', $details['order_id'])->lockForUpdate()->first();

                if($order){
                    $order['status'] = 1;
                    $order['departure_id'] = $departure['id'];
                    $order['did_no'] = $generated_no;
                    $order->update();
                }

                $details['did_no'] = $generated_no;
                $details['yard_id'] = 0;
                $details['block_id'] = 0;
                $details['status'] = 'completed';
                $details->update();

            }
        }

        // $trip_service = new TripService;
        // $trip_service->createTripDeparture($departure, $container_details);

        $response=[
            'success' => true,
            'message' => 'Departure approved successfully',
            'data' => [
                'departure' => $departure,
                'container_details' => $container_details
            ]
        ];
        return response()->json($response, 200);
    }
    public function rejectDeparture(Request $request)
    {
        // Validate the request
        $validator = validator($request->all(), [
            'id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 200);
        }

        // Find the departure record by ID and reject it
        $departure = PlacedDepartureOrder::where('id', $request['id'])->first();

        if ($departure) {
            $departure['approved'] = 0;
            $departure['status'] = 0;
            $departure['reject_reason'] = $request['reject_reason'] ?? null;
            $departure->update();

            return response()->json([
                'success' => true,
                'message' => 'Departure rejected successfully',
                'data' => $departure,
            ], 200);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Departure record not found',
            ], 404);
        }
    }

    private function generateRandomPID()
    {

        $unique = false;

        while (!$unique) {
            $random_string = 'DID' . rand(10000, 99999);
            $count = ContainerDetail::where('did_no', $random_string)->count();
            if ($count == 0) {
                $unique = true;
            }
        }

        return $random_string;
    }

    public function filter(Request $request)
    {
        $pre_arrival = PlacedDepartureOrder::where('approved', 0)->where('order_type', 'arrival')
            ->with('orders', 'orders.container_owner:id,company_name');

        if ($request['date'] != '' && isset($request['date'])) {
            $pre_arrival = $pre_arrival->whereDate('date', $request['date']);
        }
        if ($request['container_id'] != '' && isset($request['container_id'])) {
            $pre_arrival = $pre_arrival->where('container_id', $request['container_id']);
        }
        if ($request['container_status'] != '' && isset($request['container_status'])) {
            // $pre_arrival=$pre_arrival->where('container_status',$request['container_status']);
            $pre_arrival = $pre_arrival->whereHas('orders', function ($query) use ($request) {
                $query->where('container_status', $request['container_status']);
            });
        }
        if ($request['container_condition'] != '' && isset($request['container_condition'])) {
            $pre_arrival = $pre_arrival->whereHas('orders', function ($query) use ($request) {
                $query->where('container_condition', $request['container_condition']);
            });
        }

        $pre_arrival = $pre_arrival->get();

        return view('pre-departure.index', compact('pre_arrival'));
    }

}
