<?php

namespace App\Http\Controllers\Api\cms;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\cms\PreArrivalDeparture;
use App\Models\cms\PlacedDepartureOrder;
use App\Models\cms\Order;
use App\Models\cms\ContainerOwner;
use App\Models\cms\ContainerDetail;
use App\Models\cms\LogisticProvider;
use App\Models\cms\Masters\ContainerType;
use App\Models\cms\Masters\VehicleType;
use App\Models\cms\Yard;
use App\Service\cms\DocumentService;
use App\Service\cms\PreDepartureService;
use App\Service\cms\ContainerDetailsService;
use App\Service\cms\PreArrivalService;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use DB;

class ArrivalController extends Controller
{
    public function index(Request $request)
    {
        try {
            // Base query for 'arrival' orders with approved = 0 and status = 0
            $pre_arrival = PreArrivalDeparture::with('entrybyid')
                ->where('order_type', 'arrival')
                ->where('status', 0)
                ->where('approved', 0);


            // Filtering based on 'Custom date'
            if (!empty($request->custom_date)) {
                $pre_arrival->whereDate('date', $request->custom_date);
            }

            // Filtering based on from date to date
            if (!empty($request->fromDate) && !empty($request->toDate)) {
                $pre_arrival->whereBetween('date', [$request->fromDate, $request->toDate]);
            }

            // Searching based on 'search' keyword (for container_id)
            if (!empty($request->search)) {
                $search = $request->search;
                $pre_arrival->where(function ($query) use ($search) {
                    $query->whereExists(function ($subQuery) use ($search) {
                        $subQuery->select(DB::raw(1))
                            ->from('container_details')
                            ->whereRaw('FIND_IN_SET(container_details.id, pre_arrival_departure.container_detail_ids)')
                            ->where('container_details.container_id', 'like', '%' . $search . '%');
                    });
                });
            }

            // Counting total records
            $count = $pre_arrival->count();

            // Pagination parameters
            $noOfRec = $request->noofrec ?? 100; // Default 100 records per page
            $currentPage = $request->currentpage ?? 1; // Default page 1

            // Sorting functionality
            if (!empty($request->sort_by)) {
                $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
                $pre_arrival->orderBy($request->sort_by, $sortOrder);
            } else {
                $pre_arrival->orderBy('id', 'desc');
            }

            // Apply pagination
            $pre_arrival = $pre_arrival->skip($noOfRec * ($currentPage - 1))
                ->take($noOfRec)
                ->get();

            // Adding container details
            foreach ($pre_arrival as $container) {
                if (!empty($container->container_detail_ids)) {
                    $container_details = ContainerDetail::whereIn('id', explode(',', $container->container_detail_ids))
                        ->select('id', 'container_id')
                        ->get();

                    $container['container_details'] = $container_details;
                } else {
                    $container['container_details'] = collect();
                }
            }

            // Return JSON response with required structure
            return response()->json([
                'success' => true,
                'total' => $count,
                'data' => $pre_arrival,
            ], 200);
        } catch (\Exception $e) {
            // Handle exceptions and return an error response
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // public function edit($id)
    // {
    //     try {
    //         // Find the arrival record by ID with related documents
    //         $arrival = PreArrivalDeparture::with('documents')->findOrFail($id);

    //         // Fetch related data
    //         $container_details = ContainerDetail::whereIn('id', explode(',', $arrival->container_detail_ids))
    //             ->with(['images' => function ($query) {
    //                 $query->where('order_type', 'arrival');
    //             }])
    //             ->get();

    //         $containerOwners = ContainerOwner::where('status', 0)->get();
    //         $containertype = ContainerType::where('status', 0)->get();
    //         $logisticProviders = LogisticProvider::where('status', 0)->get();
    //         $vehicleMaster = VehicleType::where('status', 0)->get();
    //         $yards = Yard::where('status', 0)->get();

    //         // Return the response as JSON
    //         return response()->json([
    //             'success' => true,
    //             'message' => 'Data fetched successfully',
    //             'data' => [
    //                 'arrival' => $arrival,
    //                 'container_details' => $container_details,
    //                 'containerOwners' => $containerOwners,
    //                 'containertype' => $containertype,
    //                 'logisticProviders' => $logisticProviders,
    //                 'vehicleMaster' => $vehicleMaster,
    //                 'yards' => $yards,
    //             ],
    //         ], 200);
    //     } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Arrival record not found.',
    //         ], 404);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'An unexpected error occurred: ' . $e->getMessage(),
    //         ], 500);
    //     }
    // }



    public function details(Request $request)
    {
        try {
            // Validate the request
            $validator = validator($request->all(), [
                'id' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            // Get the ID from the request body
            $id = $request->input('id');

            // Fetch the arrival record with relationships (block and documents)
            $arrival = PreArrivalDeparture::with([
                'block:id,block_name','ship_details','port_details','vendor','yard','logistic_provider','vehicle','driver','items'
            ])->findOrFail($id);

            // Get container details with images filtered by 'arrival' order_type
            $container_details = ContainerDetail::whereIn('id', explode(',', $arrival->container_detail_ids))
                ->with([
                    // 'images' => function ($query) {
                    //     $query->where('order_type', 'arrival');
                    // },
                    'container_owner',
                    'container_types',
                    'logisticProvider',
                    'vehicleMaster',
                    'yard'
                ])
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Arrival details fetched successfully',
                'data' => [
                    'pre_arrival' => $arrival,
                    'container_details' => $container_details
                ]
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'arrival record not found',
            ], 404);
        } catch (\Exception $e) {
            // Handle other unexpected errors
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while retrieving data',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            // Get today's date
            $today = new Carbon();
            $request['date'] = $today->format('Y-m-d');

            // Creation of Services
            $documentService = new DocumentService;
            $containerDetailService = new ContainerDetailsService;

            $orders = [];
            $prearrival = null; // Initialize prearrival variable
            $arrived = null; // Initialize arrived variable

            // Pre Arrival order Type
            if ($request['order_type'] == 'pre_arrival') {
                $order_container_id = $containerDetailService->createOrderContainer($request);

                $prearrival = new PreArrivalService;
                $prearrival = $prearrival->savePreArrival($request);
                $prearrival['container_detail_ids'] = implode(',', $order_container_id);
                $prearrival->update();

                // Save Documents
                $documentService->addDocument($request, $prearrival['id'], 'pre_arrival');

                $orders[] = $prearrival;
            }

            // Arrival Order Type
            if ($request['order_type'] == 'arrival') {
                if (!isset($request['order_container_id'])) {
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                }

                $arrived = new PreArrivalService;
                $arrived = $arrived->saveArrival($request);
                $arrived['container_detail_ids'] = implode(',', $request['order_container_id'] ?? $order_container_id);
                $arrived->update();

                $documentService->addDocument($request, $arrived['id'], 'arrival');

                $orders[] = $arrived;
            }

            if ($request['order_type'] == 'pre_departure') {
                $pre_departure = new PreDepartureService;

                if (isset($request['order_container_id'])) {
                    $order_container_id = $containerDetailService->updateContainerDetail($request);

                    foreach ($request['order_container_id'] as $key => $value) {
                        $container_detail = ContainerDetail::where('id', $request['order_container_id'][$key])->first();
                        $container_detail['status'] = "ongoing";
                        $container_detail->update();
                    }
                } else {
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                }

                $pre_departure = $pre_departure->savePreDeparture($request);
                $pre_departure['container_detail_ids'] = implode(',', $request['order_container_id']);
                $pre_departure->update();

                $documentService->addDocument($request, $pre_departure['id'], 'pre_departure');

                $orders[] = $pre_departure;
            }

            if ($request['order_type'] == 'departure') {
                $departure = new PreDepartureService;

                if (isset($request['order_container_id'])) {
                    $order_container_id = $containerDetailService->updateContainerDetail($request);

                    foreach ($request['order_container_id'] as $key => $value) {
                        $container_detail = ContainerDetail::where('id', $request['order_container_id'][$key])->first();
                        $container_detail['status'] = "process";
                        $container_detail->update();
                    }
                } else {
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                }

                $departure = $departure->saveDeparture($request);
                $departure['container_detail_ids'] = implode(',', $request['order_container_id']);
                $departure->update();

                $documentService->addDocument($request, $departure['id'], 'departure');

                $orders[] = $departure;
            }

            return response()->json([
                'success' => true,
                'total' => $arrived ? 1 : 0, // Return count based on arrived data
                'data' => [
                    'arrived' => $arrived // Include the arrived data in the response
                ]
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function createtoArrival(Request $request)
    {
        try {
            // 🔹 Validate request input
            $validator = Validator::make($request->all(), [
                'type' => 'nullable|string|in:arrival,pre_arrival',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                    'data' => null,
                ], 422);
            }

            // Pagination variables
            $currentPage = $request->page ?? 1;
            $noOfRec = $request->limit ?? 50;

            // 🔹 Get Orders which are not arrived
            $orders = Order::where('status', 0)
                ->whereNull('did_no')
                ->whereHas('arrival', function ($query) {
                    $query->where('approved', 1);
                });

            // Apply filters if provided
            if (!empty($request->id) && $request->id > 0) {
                $orders->where('id', $request->id);
            }

            // Apply pagination
            $orders = $orders->skip($noOfRec * ($currentPage - 1))
                ->take($noOfRec)
                ->get();

            // 🔹 Get Pre-Arrival Orders with filtering
            $pre_arrival = PreArrivalDeparture::query();

            // Apply 'type' filter
            if (!empty($request->type)) {
                $pre_arrival->where('order_type', $request->type);
            }

            // Apply 'pending' filter
            if ($request->pending == 1) {
                $pre_arrival->where('status', 'pending');
            }

            // Apply pagination
            $pre_arrival = $pre_arrival->skip($noOfRec * ($currentPage - 1))
                ->take($noOfRec)
                ->get();

            // 🔹 Combined Container Details Array
            $all_container_details = [];

            // Attach Container Details to Each Pre-Arrival Record
            foreach ($pre_arrival as $value) {
                $container_details = ContainerDetail::whereIn('id', explode(',', $value->container_detail_ids))
                    ->with([
                        // 'images' => function ($query) {
                        //     $query->where('order_type', 'arrival'); // Fetch only relevant images
                        // },
                        'container_owner',
                        'container_types',
                        'logisticProvider',
                        'vehicleMaster',
                        'yard'
                    ])
                    ->get();


                // 🔹 Search Filter on container_id
                if (!empty($request->search)) {
                    $search = $request->search;
                    $container_details->where(function ($query) use ($search) {
                        $query->where('container_id', 'LIKE', "%{$search}%");
                    });
                }

                // Attach Container Details to Pre-Arrival Record
                $value->container_details = $container_details;

                // Merge all container details into a single array
                $all_container_details = array_merge($all_container_details, $container_details->toArray());
            }

            // ✅ Return JSON Response with Combined Container Details
            return response()->json([
                'success' => true,
                'total_container_details' => count($all_container_details),
                // 'total_pre_arrival' => count($pre_arrival),
                // 'orders' => $orders,
                // 'pre_arrival' => $pre_arrival,
                'all_container_details' => $all_container_details, // New array with all container details
                'message' => 'Pre-Arrival data fetched successfully.',
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching pre-arrival data.',
                'error' => $e->getMessage(),
                'data' => null,
            ], 500);
        }
    }







    // public function filter(Request $request)
    // {
    //     $pre_arrival = PreArrivalDeparture::where('approved', 1)->where('order_type', 'arrival')
    //         ->with('orders', 'orders.container_owner:id,company_name');

    //     if ($request['date'] != '' && isset($request['date']) && $request['date'] != null) {
    //         $pre_arrival = $pre_arrival->whereDate('date', $request['date']);
    //     }
    //     if ($request['container_id'] != '' && isset($request['container_id']) && $request['container_id'] != null) {
    //         $pre_arrival = $pre_arrival->where('container_id', $request['container_id']);
    //     }
    //     if ($request['container_status'] != '' && isset($request['container_status']) && $request['container_status'] != null) {
    //         // $pre_arrival=$pre_arrival->where('container_status',$request['container_status']);
    //         $pre_arrival = $pre_arrival->whereHas('orders', function ($query) use ($request) {
    //             $query->where('container_status', $request['container_status']);
    //         });
    //     }
    //     if ($request['container_condition'] != '' && isset($request['container_condition']) && $request['container_condition'] != null) {
    //         $pre_arrival = $pre_arrival->whereHas('orders', function ($query) use ($request) {
    //             $query->where('container_condition', $request['container_condition']);
    //         });
    //     }

    //     $pre_arrival = $pre_arrival->get();

    //     return view('pre-arrival.index', compact('pre_arrival'));
    // }


    // public function createPreDeparture(Request $request)
    // {
    //     // Validate the request
    //     // $request->validate([
    //     //     'id' => 'required|integer|exists:pre_arrival_departures,id',
    //     // ]);
    //     $validator = validator($request->all(), [
    //         'id' => 'required|integer|exists:pre_arrival_departures,id',

    //     ]);
    //     if ($validator->fails()) {
    //         return [
    //             'success' => false,
    //             'message' => $validator->errors()->first()
    //         ];
    //     } else {
    //         try {


    //             // Find the arrival record
    //             $arrival = PreArrivalDeparture::findOrFail($request->id);

    //             // Create a new pre-departure order
    //             $pre_departure = PlacedDepartureOrder::create([
    //                 'order_id' => $arrival->id,
    //                 'date' => now()->format('Y-m-d'),
    //                 'order_type' => 'pre_departure',
    //                 'container_id' => $arrival->container_id,
    //                 'logistic_provider_id' => $arrival->logistic_provider_id,
    //                 'vehicle_type' => $arrival->vehicle_type,
    //                 'vehicle_no' => $arrival->vehicle_no,
    //                 'contact_person' => $arrival->contact_person,
    //                 'phone_number' => $arrival->phone_number,
    //                 'arrival_departure_address' => $arrival->arrival_departure_address,
    //                 'client_name' => $arrival->client_name,
    //                 'client_email' => $arrival->client_email,
    //                 'client_phoneno' => $arrival->client_phoneno,
    //                 'address' => $arrival->address,
    //                 'remarks' => $arrival->remarks,
    //             ]);

    //             return response()->json([
    //                 'success' => true,
    //                 'message' => 'Order moved to Pre Departure successfully',
    //                 'data' => $pre_departure,
    //             ], 201);
    //         } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Arrival record not found',
    //             ], 404);
    //         } catch (\Exception $e) {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'An unexpected error occurred',
    //                 'error' => $e->getMessage(),
    //             ], 500);
    //         }
    //     }
    // }
}
