<?php

namespace App\Http\Controllers\Api\cms;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\cms\Order;
use App\Models\cms\Masters\ContainerType;
use App\Models\cms\ContainerDetail;
use DB;

class GatePassController extends Controller
{
    public function indexAid(Request $request)
    {
        try {
            // Base query for orders with conditions
            $orders = Order::whereNotNull('pid_no')->with('container_details');

            // Filtering based on 'pid_no'
            if (!empty($request->pid_no)) {
                $orders->where('pid_no', $request->pid_no);
            }

             // Filtering based on 'date'
            if (!empty($request->custom_date)) {
                $orders->whereDate('created_at', $request->custom_date);
            }

            // Filtering based on from date to date
            if (!empty($request->fromDate) && !empty($request->toDate)) {
                $orders->whereBetween('date', [$request->fromDate, $request->toDate]);
            }

            // Searching based on 'search' keyword (pid_no or container_id)
            if (!empty($request->search)) {
                $search = $request->search;
                $orders->where(function ($query) use ($search) {
                    $query->where('pid_no', 'like', '%' . $search . '%')
                        ->orWhere('order_id', 'like', '%' . $search . '%')
                        ->orWhereHas('container_details', function ($q) use ($search) {
                            $q->where('container_id', 'like', '%' . $search . '%');
                        });
                });
            }

            // Count total records for pagination
            $count = $orders->count();

            // Pagination parameters (with defaults)
            $noOfRec = $request->noofrec ?? 100;
            $currentPage = $request->currentpage ?? 1;

            // Sorting functionality
            if (!empty($request->sort_by)) {
                $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
                $orders->orderBy($request->sort_by, $sortOrder);
            } else {
                $orders->orderBy('id', 'desc');
            }

            // Apply pagination
            $orders = $orders->skip($noOfRec * ($currentPage - 1))
                ->take($noOfRec)
                ->get();

            // Fetch active container types
            $containertype = ContainerType::where('status', 0)->get();

            // Attach container details if needed
            foreach ($orders as $order) {
                if (!empty($order->container_detail_ids)) {
                    $container_details = ContainerDetail::whereIn('id', explode(',', $order->container_detail_ids))
                        ->select('id', 'container_id')
                        ->get();

                    $order['container_details'] = $container_details;
                } else {
                    $order['container_details'] = collect();
                }
            }

            // Response structure
            return response()->json([
                'success' => true,
                'total' => $count,
                'data' => [
                    'orders' => $orders,
                    'containertype' => $containertype,
                ],
            ], 200);
        } catch (\Exception $e) {
            // Handle unexpected errors
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    public function indexDid(Request $request)
    {
        try {
            // Base query for orders where 'did_no' is not null and 'pid_no' is not null
            $orders = Order::where('pid_no', '!=', null)
                ->whereHas('container_details', function ($q) {
                    $q->whereNotNull('did_no');
                })
                ->with('container_details','departures','departures.vendor','departures.driver','departures.vehicle','departures.block','departures.yard','departures.logistic_provider');

            // Filtering based on 'pid_no'
            if (!empty($request->pid_no)) {
                $orders->where('pid_no', $request->pid_no);
            }

            // Filtering based on 'did_no'
            if (!empty($request->did_no)) {
                $orders->whereHas('container_details', function ($q) use ($request) {
                    $q->where('did_no', $request->did_no);
                });
            }

            // Filtering based on 'date'
            if (!empty($request->custom_date)) {
                $orders->whereDate('created_at', $request->custom_date);
            }

            // Filtering based on from date to date
            if (!empty($request->fromDate) && !empty($request->toDate)) {
                $orders->whereBetween('date', [$request->fromDate, $request->toDate]);
            }

            // Searching based on 'search' keyword (pid_no or container_id)
            if (!empty($request->search)) {
                $search = $request->search;
                $orders->where(function ($query) use ($search) {
                    $query->where('pid_no', 'LIKE', "%{$search}%")
                        ->orWhere('order_id', 'LIKE', "%{$search}%")
                        ->orWhereHas('container_details', function ($q) use ($search) {
                            $q->where('container_id', 'LIKE', "%{$search}%");
                        });
                });
            }

            // Count total records for pagination
            $count = $orders->count();

            // Pagination parameters (with defaults)
            $noOfRec = $request->noofrec ?? 100;
            $currentPage = $request->currentpage ?? 1;

            // Sorting functionality
            if (!empty($request->sort_by)) {
                $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
                $orders = $orders->orderBy($request->sort_by, $sortOrder);
            } else {
                $orders = $orders->orderBy('id', 'desc');
            }

            // Apply pagination
            $orders = $orders->skip($noOfRec * ($currentPage - 1))
                ->take($noOfRec)
                ->get();

            // Attach additional container details if needed
            foreach ($orders as $order) {
                if (!empty($order->container_detail_ids)) {
                    $container_details = ContainerDetail::whereIn('id', explode(',', $order->container_detail_ids))
                        ->select('id', 'container_id', 'did_no')
                        ->get();

                    $order['container_details'] = $container_details;
                } else {
                    $order['container_details'] = collect();
                }
            }

            // Fetch active container types
            $containertype = ContainerType::where('status', 0)->get();

            // Response structure
            return response()->json([
                'success' => true,
                'total' => $count,
                'orders' => $orders
            ], 200);
        } catch (\Exception $e) {
            // Handle unexpected errors
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
