<?php

namespace App\Http\Controllers\Api\cms\Masters;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\cms\Masters\ContainerCondition;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class ContainerConditionController extends Controller
{

    // public function index(Request $request)
    // {
    //     //
    //     $type = ContainerCondition::orderBy('id', 'DESC')->get();

    //     return view('masters.container_condition', compact('type'));
    // }


    public function index(Request $request)
    {
        $type = ContainerCondition::query();

        // Filtering based on 'id'
        if (!empty($request->id) && $request->id > 0) {
            $type->where('id', $request->id);
        }

        // Filtering based on 'status' (only 0 or 1)
        if (isset($request->status) && in_array($request->status, [0, 1])) {
            $type->where('status', $request->status);
        }

        // Searching based on 'search' keyword
        if (!empty($request->search)) {
            $search = $request->search;
            $type->where(function ($query) use ($search) {
                $query->where('id', 'LIKE', "%{$search}%")
                    ->orWhere('name', 'LIKE', "%{$search}%")
                    ->orWhere('status', 'LIKE', "%{$search}%");
            });
        }

        // Counting total records
        $count = $type->count();

        // Sorting based on 'sort_by' and 'sort_order'
        if (isset($request->sort_by) && !empty($request->sort_by)) {
            $sortOrder = $request->sort_order == 1 ? 'asc' : 'desc';
            $type->orderBy($request->sort_by, $sortOrder);
        } else {
            $type->orderBy('id', 'desc');
        }

        // Pagination with 'noofrec' and 'currentpage'
        $noOfRec = $request->noofrec ?? 100;
        $currentPage = $request->currentpage ?? 1;

        $type = $type->skip($noOfRec * ($currentPage - 1))
            ->take($noOfRec)
            ->get();

        // Response structure
        return response()->json([
            'success' => true,
            'total' => $count,
            'data' => $type,
        ]);
    }



    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Validate the request
        $validator = validator($request->all(), [
            'name' => 'required|string|unique:container_conditions,name',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 400);
        }

        try {
            // Check for duplicate entry
            $duplicateCheck = ContainerCondition::where('name', $request->input('name'))->first();

            if ($duplicateCheck) {
                return response()->json([
                    'success' => false,
                    'message' => 'Container Condition with name ' . $request->input('name') . ' already exists.',
                ], 409);
            }

            // Create new Container Condition
            $type = ContainerCondition::create(['name' => $request->input('name')]);

            // Prepare response
            return response()->json([
                'success' => true,
                'message' => 'Container Condition added successfully.',
                'data' => $type,
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while adding the Container Condition: ' . $e->getMessage(),
            ], 500);
        }
    }



   public function update(Request $request)
    {
        // Validate the request (only for 'id' and 'name')
        $validator = validator($request->all(), [
            'id' => 'required|integer',
            'name' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 400);
        }

        try {
            // Find the record by ID
            $type = ContainerCondition::findOrFail($request->input('id'));

            // Update name without checking for duplicates
            $type->update(['name' => $request->input('name')]);

            // Prepare response
            return response()->json([
                'success' => true,
                'message' => 'Type updated successfully.',
                'data' => $type,
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid ID to update.',
            ], 404);
        }
    }

    public function destroy(Request $request)
    {
        // Validate the request
        $validator = validator($request->all(), [
            'id' => 'required|exists:container_conditions,id', // Ensure ID exists in the table
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ]);
        }

        try {
            // Find the ContainerCondition by ID or fail
            $type = ContainerCondition::findOrFail($request->input('id'));

            // Delete the record
            $type->delete();

            return response()->json([
                'success' => true,
                'message' => 'Type deleted successfully.',
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid ID to delete.',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while deleting the type.',
            ], 500);
        }
    }
}
