<?php

namespace App\Http\Controllers\Api\cms;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Models\cms\Order;
use App\Models\cms\ContainerOwner;
use App\Models\cms\Masters\ContainerType;
use App\Models\cms\LogisticProvider;
use App\Models\cms\Masters\VehicleType;
use App\Models\cms\Yard;
use App\Models\cms\PreArrivalDeparture;
use App\Models\cms\ContainerDetail;
use App\Models\cms\PlacedDepartureOrder;
use App\Models\cms\CmsPurchaseOrder;
use App\Models\cms\CmsItemDetail;
use App\Models\cms\PortDetail;
use App\Models\cms\ShipDetail;
// use App\Models\cms\Document;
// use App\Service\cms\DocumentService;
use App\Service\cms\PreArrivalService;
use App\Service\cms\PreDepartureService;
use App\Service\cms\OrderService;
use App\Service\cms\ContainerDetailsService;
use App\Service\cms\CmsItemDetailService;
use App\Service\cms\PortDetailService;
use App\Service\cms\ShipDetailService;
use App\Service\SequenceService;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Validator;
use Auth;
use DB;
use Log;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $orders = Order::with(
            'container_owner:id,company_name',
            'container_types:id,type',
            'departures',
            'arrival',
            'entrybyid'
        );

        // Filtering based on 'id', 'order_id', 'arrival_id', 'departure_id', 'container_id'
        if (!empty($request->id) && $request->id > 0) {
            $orders->where('id', $request->id);
        }

        if (!empty($request->order_id)) {
            $orders->where('order_id', $request->order_id);
        }

        // Filtering By Status
        if (isset($request->status)) {
            $orders->where('status', $request->status);
        }else{
            $orders->where('status', 0);
        }

        // Filtering based on 'Custom date'
        if (!empty($request->custom_date)) {
            $orders->whereDate('date', $request->custom_date);
        }

        // Filtering based on from date to date
        if (!empty($request->fromDate) && !empty($request->toDate)) {
            $orders->whereBetween('date', [$request->fromDate, $request->toDate]);
        }

        // Sorting functionality
        if (!empty($request->sort_by)) {
            $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
            $orders = $orders->orderBy($request->sort_by, $sortOrder);
        } else {
            $orders = $orders->orderBy('id', 'desc');
        }

        // Searching based on 'search' keyword
        if (!empty($request->search)) {
            $search = $request->search;
            $orders->where(function ($query) use ($search) {
                $query->where('order_id', 'LIKE', "%{$search}%")
                    ->orWhere('pid_no', 'LIKE', "%{$search}%")
                    ->orWhere('did_no', 'LIKE', "%{$search}%")
                    ->orWhereHas('container_owner', function ($query) use ($search) {
                        $query->where('company_name', 'LIKE', "%{$search}%");
                    });
            });
        }

        // Counting total records
        $count = $orders->count();

        // Pagination with 'noofrec' and 'currentpage'
        $noOfRec = $request->noofrec ?? 100;
        $currentPage = $request->currentpage ?? 1;

        $orders = $orders->skip($noOfRec * ($currentPage - 1))
            ->take($noOfRec)
            ->get();

        // Response structure
        return response()->json([
            'success' => true,
            'message' => 'Order List Retrieved Successfully',
            'total' => $count,
            'data' => $orders,
        ], 200);
    }


    public function pidOrders(Request $request){

        // Validate the request
        $validator = Validator::make($request->all(), [
            'order_id' => 'required', // Ensure ID is an integer
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'data' => null
            ], 422);
        }

        $orders = Order::where('id', $request['order_id'])
            ->first();

        if (!$orders) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found',
                'data' => null
            ], 404);
        }

        // Fetch the arrival record with relationships (Orders and Arrival)
        $arrival = PreArrivalDeparture::with([
            'block:id,block_name','ship_details','port_details','vendor','yard','logistic_provider','vehicle','driver','items'
        ])->findOrFail($orders['arrival_id']);

        // Get container details with images filtered by 'arrival' order_type
        if($arrival!=null){
            $container_details = ContainerDetail::whereIn('id', explode(',', $arrival->container_detail_ids))
                ->with([
                    'container_owner',
                    'container_types',
                    'logisticProvider',
                    'vehicleMaster',
                    'yard'
                ])
                ->get();
        } else {
            $container_details = [];
        }
        
        $response = [
            'success' => true,
            'message' => 'Order details fetched successfully',
            'data' => [
                'orders' => $orders,
                'arrival' => $arrival,
                'container_details' => $container_details
            ]
        ];

        return response()->json($response, 200);
    }


    public function pidOrdersPdf($id)
    {
        $orders = Order::with([
            'arrival',
            'arrival.block:id,block_name',
            'arrival.documents',
            'container_details.images' => function ($query) {
                $query->where('order_type', 'arrival');
            }
        ])
            ->where('id', $id)
            ->whereHas('arrival', function ($query) {
                $query->where('approved', 1);
            })
            ->first();
        // dd($orders);
        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $vehicleMaster = VehicleType::where('status', 0)->get();

        $yards = Yard::where('status', 0)->get();

        return view('pdf.arrival_pdf', compact('orders', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster', 'yards'));
    }


    // Get the list of all orders Which are Arrived and Pending for Pre Departure
    // Creating api at the time of creation of pre departure
    public function pendingPreArrivedlist(Request $request) {
        $pre_arrival = PreArrivalDeparture::where('status', 0)
            ->where('order_type', 'pre_arrival')
            ->whereExists(function ($subQuery) {
                $subQuery->select(DB::raw(1))
                    ->from('container_details')
                    ->whereRaw('FIND_IN_SET(container_details.id, pre_arrival_departure.container_detail_ids)')
                    ->where('container_details.order_status', 'pre_arrival');
            });

        // Searching based on 'search' keyword ( container_id,)
        if (!empty($request->search)) {
            $search = $request->search;
            $pre_arrival->where(function ($query) use ($search) {
                $query->whereExists(function ($subQuery) use ($search) {
                    $subQuery->select(DB::raw(1))
                        ->from('container_details')
                        ->whereRaw('FIND_IN_SET(container_details.id, pre_arrival_departure.container_detail_ids)')
                        ->where('container_details.container_id', 'like', '%' . $search . '%');
                });
            });
        }

        // Sorting functionality
        if (!empty($request->sort_by)) {
            $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
            $pre_arrival = $pre_arrival->orderBy($request->sort_by, $sortOrder);
        } else {
            $pre_arrival = $pre_arrival->orderBy('id', 'desc');
        }

        // Apply pagination
        $pre_arrival = $pre_arrival->get();

        $response=[
            'success'=>true,
            'message'=>'Pending Pre Arrival Lists',
            'pre_arrival'=>$pre_arrival
        ];

        return response()->json($response);
    }
    // Get the list of all orders Which are Arrived and Pending for Pre Departure
    // Creating api at the time of creation of pre departure
    public function pendingArrivedContainers(Request $request) {
        $container_details=ContainerDetail::where('status','pending')
            ->where('order_status','pre_arrival')
            ->with('container_types','container_owner','block','yard');

        if($request['container_id']!=null && $request['container_id']!=''){
            $container_details->where('container_id',$request['container_id']);
        }
        
        $count = $container_details->count();

        $container_details=$container_details->skip($request['noofrec'] * ($request['currentpage'] - 1))
            ->orderBy('id','desc')
            ->take($request['noofrec']??10)
            ->get();

        $response=[
            'success'=>true,
            'count'=>$count,
            'message'=>'Pending Pre Arrival Containers',
            'container_details'=>$container_details
        ];

        return response()->json($response);
    }

    // Get all the list of Inventory containers
    public function inventoryContainers(Request $request) {

        $container_details=ContainerDetail::where('status','pending')
            ->whereNotNull('pid_no')
            ->whereNull('did_no')
            ->where('order_id','>', 0)
            ->where('status','!=','completed')
            ->with('container_types','container_owner');

        if($request['container_id']!=null && $request['container_id']!=''){
            $container_details->where('container_id',$request['container_id']);
        }

        $count = $container_details->count();

        $container_details=$container_details->skip($request['noofrec'] * ($request['currentpage'] - 1))
            ->take($request['noofrec']??10)
            ->orderBy('id','desc')
            ->get();

        $response=[
            'success'=>true,
            'count'=>$count,
            'message'=>'Inventory Containers List',
            'container_details'=>$container_details
        ];
        return response()->json($response);
    }

    // Get all the list of Pre Departure containers
    public function getPreDepartureContainers(Request $request)
    {
        // Fetch search parameter (if provided)
        $searchContainerId = $request->input('container_id');

        // Fetch pre_departure orders
       $pre_departure_query = PlacedDepartureOrder::where('status', 0)
            ->where('approved', 0)
            ->where('order_type', 'pre_departure');

        if($request['search']!=null && $request['search']!=''){
            $pre_departure_query->where('pre_departure_id','like','%'.$request['search'].'%');
        }

        $count = $pre_departure_query->count();

        $pre_departure = $pre_departure_query
            ->orderBy('id', 'desc')
            ->skip($request['noofrec'] * ($request['currentpage'] - 1))
            ->take($request['noofrec'] ?? 10)
            ->get();

        foreach ($pre_departure as $key => $value) {
            $containerIds = [];

            if (!empty($value['container_detail_ids'])) {
                $containerIds = explode(',', $value['container_detail_ids']);
                $containerIds = array_filter(array_unique(array_map('trim', $containerIds)));
            }

            $container_details = [];
            if (!empty($containerIds)) {
                $container_details = ContainerDetail::whereIn('id', $containerIds)
                    ->with(['container_types', 'container_owner'])
                    ->get();
            }

            $pre_departure[$key]['container_details'] = $container_details;
        }

        // Convert comma-separated container IDs into an array
        // $containerIds = [];
        // foreach ($pre_departure['container_detail_ids'] as $ids) {
        //     $containerIds = array_merge($containerIds, explode(',', $ids));
        // }

        // // Fetch container details
        // $containerQuery = ContainerDetail::whereIn('id', array_unique($containerIds))
        //     ->with('container_types','container_owner');

        // // Apply search filter if container_id is provided
        // if (!empty($searchContainerId)) {
        //     $containerQuery->where('container_id', 'LIKE', "%$searchContainerId%");
        // }

        // // Count the total number of container details
        // $count = $containerQuery->count();

        // // Get the filtered container details
        // $containerDetails = $containerQuery->skip($request['noofrec'] * ($request['currentpage'] - 1))
        //     ->orderBy('id', 'desc')
        //     ->take($request['noofrec']??10)
        //     ->get();

        $response=[
            'success'=>true,
            'count'=>$count,
            'message'=>'Pre Departure Containers List',
            // 'container_details'=>$containerDetails
            'pre_departure'=>$pre_departure
        ];

        return response()->json($response);
    }


    public function ordersCreateUpdate(Request $request) {

        // Validate Item
        $validator=validator($request->all(),[
            'order_type'=>'required|string',
        ]);

        if ($validator->fails()) { 

            return [
                'success' => false, 
                'message' => $validator->errors()->first()
            ];

        } else {

            try {
                // Get today's date
                $today = new Carbon();
                $request['date'] = $today->format('Y-m-d');
        
                // Creation of Objects
                // $documentService = new DocumentService;
                $containerDetailService = new ContainerDetailsService;
                $sequence=new SequenceService;
        
                $orders = [];
                $route = '';

                // Pre Arrival order Type
                if ($request['order_type'] == 'pre_arrival') {

                    if($request['id']>0){
                        CmsItemDetail::where('transaction_id',$request['id'])
                            ->where('order_type','pre_arrival')
                            ->delete();

                        PortDetail::where('transaction_id',$request['id'])
                        ->where('order_type','pre_arrival')
                        ->delete();
                    
                        ShipDetail::where('transaction_id',$request['id'])
                        ->where('order_type','pre_arrival')
                        ->delete();

                    }else{
                        $module='pre_arrival';
                        // Update Sequence Number
                        $sequence->updateSequence($module);
                    }

                    $container_detail_ids = $containerDetailService->createOrderContainer($request);

                    $prearrival = new PreArrivalService;
                    // Create and Update Pre Arrival Order
                    $prearrival = $prearrival->savePreArrival($request);

                    // Use object property syntax
                    $prearrival->container_detail_ids = implode(',', $container_detail_ids);
                    $prearrival->vendor_id = $request['vendor_id'] ?? 0;

                    $prearrival->update();

                    // Creating Item 
                    $itemservice=new CmsItemDetailService;
                    $itemservice->itemCreateUpdate($request,$prearrival);

                    // Creating Port 
                    $portService = new PortDetailService;
                    $portService->portCreateUpdate($request,$prearrival);

                    // Ship Details
                    $portService = new ShipDetailService;
                    $portService->shipCreateUpdate($request,$prearrival);
        
                    // Save Documents
                    // $documentService->addDocument($request, $prearrival['id'], 'pre_arrival');
        
                    $orders[] = $prearrival;
                }
        
                // Arrival order Type
                if ($request['order_type'] == 'arrival') {

                    if ($request['id']>0) {

                        CmsItemDetail::where('transaction_id',$request['id'])
                        ->where('order_type','arrival')
                        ->delete();

                        PortDetail::where('transaction_id',$request['id'])
                        ->where('order_type','arrival')
                        ->delete();
                    
                        ShipDetail::where('transaction_id',$request['id'])
                        ->where('order_type','arrival')
                        ->delete();

                    }else{
                        $module='arrival';
                        // Update Sequence Number
                        $sequence->updateSequence($module);
                    }

                    $container_detail_ids = $containerDetailService->createOrderContainer($request);
        
                    $arrived = new PreArrivalService;
                    $arrived = $arrived->saveArrival($request);
                    $arrived['container_detail_ids'] = implode(',', $request['container_detail_ids'] ?? $container_detail_ids);
                    $arrived->update();

                    // Creating Item 
                    $itemservice=new CmsItemDetailService;
                    $itemservice->itemCreateUpdate($request,$arrived);

                    // Creating Port 
                    $portService = new PortDetailService;
                    $portService->portCreateUpdate($request,$arrived);

                    // Ship Details
                    $portService = new ShipDetailService;
                    $portService->shipCreateUpdate($request,$arrived);
        
                    // Save Documents
                    // $documentService->addDocument($request, $arrived['id'], 'arrival');
        
                    $orders[] = $arrived;
                }
        
                // Pre Departure Order type
                if ($request['order_type'] == 'pre_departure') {

                    if ($request['id']>0) {

                        CmsItemDetail::where('transaction_id',$request['id'])
                        ->where('order_type','pre_departure')
                        ->delete();

                    }else{
                        $module='pre_departure';
                        // Update Sequence Number
                        $sequence->updateSequence($module);
                    }

                    $pre_departure = new PreDepartureService;
                  
                    $container_detail_ids = $containerDetailService->createOrderContainer($request);
                    
                    $pre_departure = $pre_departure->savePreDeparture($request);
                    $pre_departure['container_detail_ids'] = implode(',', $request['container_detail_ids'] ?? $container_detail_ids);
                    $pre_departure->update();

                    // Creating Item 
                    $itemservice=new CmsItemDetailService;
                    $itemservice->itemCreateUpdate($request,$pre_departure);
        
                    $orders[] = $pre_departure;

                }
        
                // Departure Order type
                if ($request['order_type'] == 'departure') {

                     if ($request['id']>0) {

                        CmsItemDetail::where('transaction_id',$request['id'])
                        ->where('order_type','departure')
                        ->delete();

                    }else{
                        $module='departure';
                        // Update Sequence Number
                        $sequence->updateSequence($module);
                    }

                    $departure = new PreDepartureService;
        
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                    
                    $departure = $departure->saveDeparture($request);
                    $departure['container_detail_ids'] = implode(',', $request['order_container_id'] ?? $order_container_id);
                    $departure->update();
        
                    $orders[] = $departure;

                    //If the record is selected from the pre departure
                    if ($request['pre_departure_id'] > 0) {
                        $pre_departure = PlacedDepartureOrder::find($request['pre_departure_id']);
                        $pre_departure['status'] = 1;
                        $pre_departure->update();
                    }

                    // Creating Item 
                    $itemservice=new CmsItemDetailService;
                    $itemservice->itemCreateUpdate($request,$departure);

                }
        
                return response()->json([
                    'success' => true,
                    'message' => 'Order created successfully',
                    'total' => count($orders),
                    'data' => [
                        'orders' => $orders
                    ]
                ], 200);
            } 
            catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error occurred while processing the request',
                    'error' => $e->getMessage()
                ], 500);
            }
        }
    }

    public function filter(Request $request)
    {
        $orders = Order::with('container_owner:id,company_name', 'container_types:id,type')
            ->where('pid_no', '!=', null);

        if ($request['order_id'] != '' && isset($request['order_id']) && $request['order_id'] != 0) {
            $orders = $orders->where('id', $request['order_id']);
        }
        if ($request['date'] != '' && isset($request['date'])) {
            $orders = $orders->whereDate('date', $request['date']);
        }
        if ($request['container_type'] != '' && isset($request['container_type'])) {
            $orders = $orders->where('container_type', $request['container_type']);
        }
        if ($request['container_owner'] != '' && isset($request['container_owner'])) {
            $orders = $orders->whereHas('container_owner', function ($query) use ($request) {
                $query->where('company_name', $request['container_owner']);
            });
        }
        if ($request['container_status'] != '' && isset($request['container_status'])) {
            // $pre_arrival=$pre_arrival->where('container_status',$request['container_status']);
            $orders = $orders->where('container_status', $request['container_status']);
        }
        if ($request['container_condition'] != '' && isset($request['container_condition'])) {
            $orders = $orders->where('container_condition', $request['container_condition']);
        }
        if ($request['order_status'] != '' && isset($request['order_status'])) {
            $orders = $orders->where('status', $request['order_status']);
        }

        $orders = $orders->get();

        $containertype = ContainerType::where('status', 0)->get();

        $currentRoute = \Route::currentRouteName();

        return view('orders.index', compact('orders', 'containertype', 'currentRoute'));
    }

    public function fetchOrderByid(Request $request)
    {

        if (isset($request['container_id'])) {
            $container_details = ContainerDetail::whereIn('id', $request['container_id'])->get();
            // dd($container_details);
        }

        $type = $request['type'];

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $vehicleMaster = VehicleType::where('status', 0)->get();

        // return response()->json(['orders'=>$orders,'container_details'=>$container_details]);
        return view('orders.container_details_row', compact('container_details', 'logisticProviders', 'containerOwners', 'containertype', 'vehicleMaster', 'type'))->render();
    }


    public function pidOrderGatepassPdf($id)
    {
        $orders = Order::with([
            'arrival',
            'arrival.block:id,block_name',
            'arrival.documents',
            'container_details.images' => function ($query) {
                $query->where('order_type', 'arrival');
            }
        ])
            ->where('id', $id)
            ->whereHas('arrival', function ($query) {
                $query->where('approved', 1);
            })
            ->first();
        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $vehicleMaster = VehicleType::where('status', 0)->get();

        return view('pdf.arrival_gatepass', compact('orders', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster'));
    }


    public function didOrdersPdf($id)
    {
        $orders = Order::with([
            'departures',
            'departures.documents',
            'container_details.images' => function ($query) {
                $query->where('order_type', 'departure');
            }
        ])
            ->where('id', $id)
            ->whereHas('departures', function ($query) {
                $query->where('approved', 1);
            })
            ->first();

        // dd($orders);
        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $vehicleMaster = VehicleType::where('status', 0)->get();

        return view('pdf.departure_pdf', compact('orders', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster'));
    }

    public function didOrderGatepassPdf($id)
    {
        $orders = Order::with([
            'departures',
            'departures.documents',
            'container_details.images' => function ($query) {
                $query->where('order_type', 'departure');
            }
        ])
            ->where('id', $id)
            ->whereHas('departures', function ($query) {
                $query->where('approved', 1);
            })
            ->first();
        // dd($orders);
        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $vehicleMaster = VehicleType::where('status', 0)->get();

        $yards = Yard::where('status', 0)->get();

        return view('pdf.departure_gatepass', compact('orders', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster', 'yards'));
    }

    public function didOrders(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'order_id' => 'required', // Ensure ID is an integer
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'data' => null
            ], 422);
        }

        $orders = Order::with([
            'departures'
        ])
            ->where('id', $request['order_id'])
            ->whereHas('departures', function ($query) {
                $query->where('approved', 1);
            })
            ->first();

        if (!$orders) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found',
            ], 404);
        }

          // Fetch the arrival record with relationships (Orders and Arrival)
          $departure = PlacedDepartureOrder::with([
            'block','vendor','yard','logistic_provider','vehicle','driver'
        ])->findOrFail($orders['departure_id']);

        // Get container details with images filtered by 'departure' order_type
        if($departure!=null){
            $container_details = ContainerDetail::whereIn('id', explode(',', $departure->container_detail_ids))
                ->with([
                    'container_owner',
                    'container_types',
                    'logisticProvider',
                    'vehicleMaster',
                    'yard'
                ])
                ->get();
        } else {
            $container_details = [];
        }
        
        $response = [
            'success' => true,
            'message' => 'Order details fetched successfully',
            'data' => [
                'orders' => $orders,
                'departure' => $departure,
                'container_details' => $container_details
            ]
        ];

        return response()->json($response, 200);

    }


    // this function call when container selected at the time of arrival , we pass container id
    public function getArrivalOrder(Request $request)
    {
        $pre_arrival = PreArrivalDeparture::where('order_type', 'pre_arrival')
            ->where('status', 0)
            ->where('approved', 0)
            ->get();

        $pre_arrival_id = [];

        if (count($pre_arrival) > 0) {
            foreach ($pre_arrival as $value) {
                if (Str::contains($value['container_detail_ids'], $request['container_id'])) {
                    $pre_arrival_id = $value;
                    break;
                } else {
                    $pre_arrival_id = null;
                }
            }
        }
        if ($pre_arrival_id != null) {
            if ($pre_arrival_id['status'] == 0) {
                $response = [
                    'success' => true,
                    'arrival_id' => $pre_arrival_id['id'],
                    'msg' => 'Container Added Successfully'
                ];
            } else {
                $response = [
                    'success' => false,
                    'arrival_id' => $pre_arrival_id['id'],
                    'msg' => 'Container Already Added in Arrival'
                ];
            }
        } else {
            $response = [
                'success' => false,
                'arrival_id' => null,
                'msg' => 'Container Already Added in Arrival'
            ];
        }

        return response()->json($response);
    }


}
