<?php

namespace App\Http\Controllers\Api\cms;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\cms\PlacedDepartureOrder;
use App\Models\cms\Order;
use App\Models\cms\ContainerOwner;
use App\Models\cms\ContainerDetail;
use App\Models\cms\LogisticProvider;
use App\Models\cms\Document;
use App\Models\cms\Masters\ContainerType;
use App\Models\cms\Masters\VehicleType;
use App\Models\cms\Yard;
use App\Service\cms\ImageService;
use App\Service\cms\DocumentService;
use App\Service\cms\PreArrivalService;
use App\Service\cms\ContainerDetailsService;
use App\Service\cms\PreDepartureService;
use Illuminate\Support\Facades\Session;
use DB;
use Carbon\Carbon;



class PreDepartureController extends Controller
{
    public function index(Request $request)
    {
        try {
            // Base query for 'pre_departure' orders with status = 0
            $pre_departure = PlacedDepartureOrder::where('status', 0)
                ->where('order_type', 'pre_departure')
                ->with('orders', 'entrybyid');

            // Filtering by 'container_id'
            if (!empty($request->container_id)) {
                $pre_departure->where('container_id', $request->container_id);
            }

            if (!empty($request->custom_date)) {
                $pre_departure->whereDate('date', $request->custom_date);
            }

             // Filtering based on from date to date
            if (!empty($request->fromDate) && !empty($request->toDate)) {
                $pre_departure->whereBetween('date', [$request->fromDate, $request->toDate]);
            }

            // Searching based on 'search' keyword ( container_id,)
            if (!empty($request->search)) {
                $search = $request->search;
                $pre_departure->where(function ($query) use ($search) {
                    $query->whereExists(function ($subQuery) use ($search) {
                        $subQuery->select(DB::raw(1))
                            ->from('container_details')
                            ->whereRaw('FIND_IN_SET(container_details.id, placed_departure_order.container_detail_ids)')
                            ->where('container_details.container_id', 'like', '%' . $search . '%');
                    });
                });
            }

            // Counting total records
            $count = $pre_departure->count();

            // Pagination parameters
            $noOfRec = $request->noofrec ?? 100;
            $currentPage = $request->currentpage ?? 1;

            // Sorting functionality
            if (!empty($request->sort_by)) {
                $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
                $pre_departure = $pre_departure->orderBy($request->sort_by, $sortOrder);
            } else {
                $pre_departure = $pre_departure->orderBy('id', 'desc');
            }

            // Pagination logic
            $pre_departure = $pre_departure->skip($noOfRec * ($currentPage - 1))
                ->take($noOfRec)
                ->get();

            // Adding container details
            foreach ($pre_departure as $departure) {
                if (!empty($departure->container_detail_ids)) {
                    // Fetch container details based on IDs
                    $container_details = ContainerDetail::whereIn('id', explode(',', $departure->container_detail_ids))
                        ->select('id', 'container_id')
                        ->get();

                    // Attach container details to the departure object
                    $departure['container_details'] = $container_details;
                } else {
                    $departure['container_details'] = collect();
                }
            }

            // Return JSON response
            return response()->json([
                'success' => true,
                'total' => $count,
                'data' => $pre_departure,
            ], 200);
        } catch (\Exception $e) {
            // Handle errors
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    public function details(Request $request)
    {
        try {
            // Validate the request
            $validator = validator($request->all(), [
                'id' => 'required',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            // Fetch the pre-departure record with related orders and documents
            $pre_departure = PlacedDepartureOrder::with('orders','logistic_provider','vendor','driver','vehicle','items')->findOrFail($request->id);

            // Get container details          
                $container_details = ContainerDetail::whereIn('id', explode(',', $pre_departure->container_detail_ids))
                ->with([
                    // 'images' => function ($query) {
                    //     $query->where('order_type', 'pre_departure');
                    // },
                    'container_owner',
                    'container_types',
                    'logisticProvider',
                    'vehicleMaster',
                    'yard'
                ])
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'pre_Arrival details fetched successfully',
                'data' => [
                    'pre_arrival' => $pre_departure,
                    'container_details' => $container_details
                ]
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // Record not found error
            return response()->json([
                'success' => false,
                'message' => 'Pre-departure record not found',
            ], 404);
        } catch (\Exception $e) {
            // Unexpected error handling
            return response()->json([
                'success' => false,
                'message' => 'An unexpected error occurred',
                'error' => $e->getMessage(),
            ], 500);
        }
    }


    public function store(Request $request)
    {
        try {
            // Get today's date
            $today = new Carbon();
            $request['date'] = $today->format('Y-m-d');

            // Creation of Services
            // $documentService = new DocumentService;
            $containerDetailService = new ContainerDetailsService;

            $orders = [];
            $prearrival = null;
            $arrived = null;
            $pre_departure = null; // Initialize pre_departure variable

            // Pre Arrival Order Type
            if ($request['order_type'] == 'pre_arrival') {
                $order_container_id = $containerDetailService->createOrderContainer($request);

                $prearrival = new PreArrivalService;
                $prearrival = $prearrival->savePreArrival($request);
                $prearrival['container_detail_ids'] = implode(',', $order_container_id);
                $prearrival->update();

                // Save Documents
                // $documentService->addDocument($request, $prearrival['id'], 'pre_arrival');

                $orders[] = $prearrival;
            }

            // Arrival Order Type
            if ($request['order_type'] == 'arrival') {
                if (!isset($request['order_container_id'])) {
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                }

                $arrived = new PreArrivalService;
                $arrived = $arrived->saveArrival($request);
                $arrived['container_detail_ids'] = implode(',', $request['order_container_id'] ?? $order_container_id);
                $arrived->update();

                // $documentService->addDocument($request, $arrived['id'], 'arrival');

                $orders[] = $arrived;
            }

            // Pre-Departure Order Type
            if ($request['order_type'] == 'pre_departure') {
                $pre_departure = new PreDepartureService;

                if (isset($request['order_container_id'])) {
                    $order_container_id = $containerDetailService->updateContainerDetail($request);

                    foreach ($request['order_container_id'] as $key => $value) {
                        $container_detail = ContainerDetail::where('id', $request['order_container_id'][$key])->first();
                        $container_detail['status'] = "ongoing";
                        $container_detail->update();
                    }
                } else {
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                }

                $pre_departure = $pre_departure->savePreDeparture($request);
                $pre_departure['container_detail_ids'] = implode(',', $request['order_container_id']);
                $pre_departure->update();

                // $documentService->addDocument($request, $pre_departure['id'], 'pre_departure');

                $orders[] = $pre_departure;
            }

            // Departure Order Type
            if ($request['order_type'] == 'departure') {
                $departure = new PreDepartureService;

                if (isset($request['order_container_id'])) {
                    $order_container_id = $containerDetailService->updateContainerDetail($request);

                    foreach ($request['order_container_id'] as $key => $value) {
                        $container_detail = ContainerDetail::where('id', $request['order_container_id'][$key])->first();
                        $container_detail['status'] = "process";
                        $container_detail->update();
                    }
                } else {
                    $order_container_id = $containerDetailService->createOrderContainer($request);
                }

                $departure = $departure->saveDeparture($request);
                $departure['container_detail_ids'] = implode(',', $request['order_container_id']);
                $departure->update();

                // $documentService->addDocument($request, $departure['id'], 'departure');

                $orders[] = $departure;
            }

            return response()->json([
                'success' => true,
                'total' => $pre_departure ? 1 : 0, // Return count based on pre_departure data
                'data' => [
                    'pre_departure' => $pre_departure // Include pre_departure data in response
                ]
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error occurred while processing the request',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function edit($id)
    {
        $pre_departure = PlacedDepartureOrder::with('orders')->where('id', $id)
            ->first();
        // dd($logisticProvider);

        $container_details = ContainerDetail::whereIn('id', explode(',', $pre_departure['container_detail_ids']))->get();

        $containerOwners = ContainerOwner::where('status', 0)->get();

        $containertype = ContainerType::where('status', 0)->get();

        $logisticProviders = LogisticProvider::where('status', 0)->get();

        $vehicleMaster = VehicleType::where('status', 0)->get();

        return view('pre-departure.edit', compact('pre_departure', 'containerOwners', 'containertype', 'logisticProviders', 'vehicleMaster', 'container_details'));
    }

    public function moveToDeparture(Request $request)
    {
        $validator=validator($request->all(),[
            'id'=>'required'
        ]);

        //Inout 1 = payment rec , 2= payment made

        if ($validator->fails()) { 
            return [
                'success' => false, 
                'message' => $validator->errors()->first()
            ];
        } else {

            $today = new Carbon();
            $request['date'] = $today->format('Y-m-d');

            $pre_departure = PlacedDepartureOrder::where('id', $request['id'])
                ->first();

            // $request['order_id']=$pre_departure['order_id'];

            $departureservice = new PreDepartureService;
            $departure = $departureservice->saveDeparture($request);
            $departure['container_detail_ids'] = implode(',', $request['order_container_id']);
            $departure->update();

            // Update Containers
            $containerDetailService = new ContainerDetailsService;
            $containerDetailService->updateContainerDetail($request);

            if (isset($request['order_container_id'])) {
                foreach ($request['order_container_id'] as $key => $value) {
                    $container_detail = ContainerDetail::where('id', $request['order_container_id'][$key])->first();
                    $container_detail['status'] = "process";
                    $container_detail->update();
                }
            }

            $pre_departure['status'] = 1;
            $pre_departure->update();

            if (isset($request['inspection']) && $request['inspection'] == 1) {
                // return redirect()->route('dashboard.home')->with('success', 'Order Moved to Departure Successfully');
            }

            $response=[
                'success' => true,
                'message' => 'Order Moved to Departure Successfully',
                'data' => [
                    'departure' => $departure
                ]
            ];
            return response()->json($response, 200);

        }
    }

    public function backTodeparture(Request $request, $id)
    {

        $arrived = PlacedDepartureOrder::where('id', $id)
            ->where('order_type', 'departure')->first();

        $arrived['status'] = 0;
        $arrived['reject_reason'] = $request['reason'];
        $arrived->update();

        // set session
        Session::flash('success', 'Order Rejected Successfully');

        return redirect()->route('departures.index')->with('success', 'Order Moved to Departure Successfully');
    }

    public function canclePreOrder(Request $request)
    {

        $validator=validator($request->all(),[
            'id'=>'required'
        ]);

        //Inout 1 = payment rec , 2= payment made

        if ($validator->fails()) { 
            return [
                'success' => false, 
                'message' => $validator->errors()->first()
            ];
        } else {
            $pre_departure = PlacedDepartureOrder::where('id', $request['id'])->first();

            // Roll Back Container Details
            $container_details = ContainerDetail::whereIn('id', explode(',', $pre_departure['container_detail_ids']))->get();
            foreach ($container_details as $key => $value) {
                $value['status'] = 'pending';
                $value['order_status'] = 'arrival';
                $value->update();
            }

            $pre_departure->delete();

            $response = [
                'success' => true,
                'message' => 'Pre Departure Order Cancelled Successfully',
            ];
            return response()->json($response, 200);
        }
    }
}
