<?php

namespace App\Http\Controllers\Api\cms;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\cms\Yard;
use App\Models\cms\Block;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class YardController extends Controller
{

  public function index(Request $request)
    {
        $yards = Yard::with('entrybyid:id,name');

        // Filtering based on 'id'
        if (!empty($request->id) && $request->id > 0) {
            $yards->where('id', $request->id);
        }

        // Filtering based on 'yard_id'
        if (!empty($request->yard_id)) {
            $yards->where('yard_id', $request->yard_id);
        }

        // Filtering based on 'status' (only 0 or 1)
        if (isset($request->status) && in_array($request->status, [0, 1])) {
            $yards->where('status', $request->status);
        }

        // Searching based on 'search' keyword
        if (!empty($request->search)) {
            $search = $request->search;
            $yards->where(function ($query) use ($search) {
                $query->where('yard_name', 'LIKE', "%{$search}%")
                    ->orWhere('location', 'LIKE', "%{$search}%")
                    ->orWhere('yard_id', 'LIKE', "%{$search}%")
                    ->orWhereHas('entrybyid', function ($query) use ($search) {
                        $query->where('name', 'LIKE', "%{$search}%");
                    });
            });
        }

        // Counting total records
        $count = $yards->count();

        // Pagination parameters
        $noOfRec = $request->noofrec ?? 100;
        $currentPage = $request->currentpage ?? 1;

        // Sorting functionality
        if (!empty($request->sort_by)) {
            $sortOrder = ($request->sort_order == 1) ? 'asc' : 'desc';
            $yards = $yards->orderBy($request->sort_by, $sortOrder);
        } else {
            $yards = $yards->orderBy('id', 'desc');
        }           

        // Pagination
        $yards = $yards->skip($noOfRec * ($currentPage - 1))
            ->take($noOfRec)
            ->get();

        // Response structure
        return response()->json([
            'success' => true,
            'total' => $count,
            'data' => $yards,
        ]);
    }


    // public function store(Request $request)
    // {
    //     $user = Auth::user();

    //     // Validate the request
    //     $validator = validator($request->all(), [
    //         'yard_name' => 'required|string',
    //         'location' => 'required|string',
    //     ]);

    //     if ($validator->fails()) {
    //         return [
    //             'success' => false,
    //             'message' => $validator->errors()->first(),
    //         ];
    //     } else {
    //         try {
    //             // Check for duplicate Yard
    //             $duplicateCheck = Yard::where('yard_name', $request['yard_name'])
    //                 ->where('location', $request['location'])
    //                 ->first();

    //             if ($duplicateCheck) {
    //                 return response()->json([
    //                     'success' => false,
    //                     'message' => 'Duplicate Yard with name ' . $request['yard_name'] . ' and location ' . $request['location'] . ' already exists.',
    //                 ], 409);
    //             }

    //             // Create new Yard
    //             $yard = Yard::create($request->all());

    //             // Generate yard ID
    //             $yard['yard_id'] = 'YD00' . $yard['id'];
    //             $yard->update();

    //             // Prepare response
    //             $response = [
    //                 'success' => true,
    //                 'message' => 'Yard Created Successfully',
    //                 'data' => $yard,
    //             ];
    //         } catch (\Exception $e) {
    //             // Handle any unexpected errors
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'An error occurred while creating the yard: ' . $e->getMessage(),
    //             ], 500);
    //         }
    //     }

    //     return response()->json($response, 201);
    // }

    public function store(Request $request)
    {
        $user = Auth::user();

        // Validate only 'yard_name' and 'location'
        $validator = validator($request->all(), [
            'yard_name' => 'required|string',
            'location' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            // Check for duplicate Yard
            $duplicateCheck = Yard::where('yard_name', $request->yard_name)
                ->where('location', $request->location)
                ->first();

            if ($duplicateCheck) {
                return response()->json([
                    'success' => false,
                    'message' => 'Duplicate Yard with name "' . $request->yard_name . '" and location "' . $request->location . '" already exists.',
                ], 409);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while creating the yard: ' . $e->getMessage(),
            ], 500);
        }
            // Create new Yard with all request data
            $yard = Yard::create($request->all());

            // Generate and update 'yard_id'
            $yard->yard_id = 'YD00' . $yard->id;
            $yard->save();

            return response()->json([
                'success' => true,
                'message' => 'Yard Created Successfully',
                'data' => $yard,
            ], 201);
       
    }




    public function show(Request $request)
    {
        // Get the ID from the request body
        $id = $request->input('id');

        // Find the yard by ID
        $yards = Yard::with('city:id,name','state:id,name','country:id,name')->findOrFail($id);

        // Get the block count based on the yard ID
        $blockcount = Block::where('yard_id', $yards->id)
            ->where('status', 1)
            ->count();

        // Return the response as JSON
        return response()->json([
            'success' => true,
            'message' => 'success',
            'data' => [
                'yard' => $yards,
                'blockcount' => $blockcount,
            ],
        ], 200);
    }


    public function edit($id)
    {

        $yards = Yard::findOrFail($id);

        return view('yard.edit', compact('yards'));
    }

    public function update(Request $request)
    {
        $user = Auth::user();

        // Validate only id, yard_name, and location
        $validator = validator($request->all(), [
            'id' => 'required|integer|exists:yards,id',
            'yard_name' => 'required|string|max:255',
            'location' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            // Find the Yard by ID
            $yard = Yard::findOrFail($request->id);

            // Check for duplicate Yard excluding current ID
            $duplicateCheck = Yard::where('yard_name', $request->yard_name)
                ->where('location', $request->location)
                ->where('id', '!=', $request->id)
                ->exists();
                // dd(123);

            if ($duplicateCheck) {
                return response()->json([
                    'success' => false,
                    'message' => 'Duplicate Yard with yard_name ' . $request->yard_name . ' and location ' . $request->location . ' already exists.',
                ], 409);
            }

            $updateData = $request->all();
            $yard->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Yard Updated Successfully',
                'data' => $yard,
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json(['success' => false, 'message' => 'Invalid ID to update'], 404);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'An error occurred: ' . $e->getMessage()], 500);
        }
    }



    public function destroy(Request $request)
    {
        $user = Auth::user();

        // Validate the request
        $validator = validator($request->all(), [
            'id' => 'required|exists:yards,id', // Validate ID exists in Yard table
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ]);
        }

        try {
            // Find the Yard by ID or fail
            $yard = Yard::findOrFail($request->input('id'));

            // Delete the yard record
            $yard->delete();

            return response()->json([
                'success' => true,
                'message' => 'Yard deleted successfully.',
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid ID to delete.',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while deleting the yard.',
            ], 500);
        }
    }




    public function yard_status(Request $request)
    {
        // Validate the request
        $validator = validator($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first()
            ]);
        }

        try {
            // Find the Yard record by ID or throw a ModelNotFoundException
            $yard = Yard::where('id', $request->input('id'))->firstOrFail();

            // Toggle the status and set the message
            if ($yard->status == 1) {
                $yard->status = 0;
                $msg = "Inactive Successfully";
            } else {
                $yard->status = 1;
                $msg = "Active Successfully";
            }

            // Save the updated status
            $yard->save();

            return response()->json([
                'success' => true,
                'message' => $msg
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            // Return error message if the record is not found
            return response()->json([
                'success' => false,
                'message' => 'Invalid ID to update'
            ], 404);
        } catch (\Exception $e) {
            // Handle any other exceptions
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while updating the status',
                'error' => $e->getMessage()
            ], 500);
        }
    }




    public function filter(Request $request)
    {
        $yards = Yard::query();

        if ($request['status'] != '' && isset($request['status'])) {
            $yards = $yards->where('status', $request['status']);
        }
        if ($request['yard_id'] != '' && isset($request['yard_id']) && $request['yard_id'] != 0) {
            $yards = $yards->where('yard_id', $request['yard_id']);
        }
        if ($request['yard_name'] != '' && isset($request['yard_name'])) {
            $yards = $yards->where('yard_name', $request['yard_name']);
        }
        if ($request['area'] != '' && isset($request['area'])) {
            $yards = $yards->where('area', $request['area']);
        }
        if ($request['capacity'] != '' && isset($request['capacity'])) {
            $yards = $yards->where('capacity', $request['capacity']);
        }
        if ($request['location'] != '' && isset($request['location'])) {
            $yards = $yards->where('location', $request['location'])
                ->orWhere('city', $request['location']);
        }
        $yards = $yards->get();

        return response()->json([
            'success' => true,
            'message' => 'success',
            'data' => $yards
        ]);
    }
}
